// shared/types/attendance-sheet.ts

import { ConcernedTrip } from "./absence";

export enum MonthsString {
  Janvier = 'Janvier',
  Février = 'Février',
  Mars = 'Mars',
  Avril = 'Avril',
  Mai = 'Mai',
  Juin = 'Juin',
  Juillet = 'Juillet',
  Août = 'Août',
  Septembre = 'Septembre',
  Octobre = 'Octobre',
  Novembre = 'Novembre',
  Décembre = 'Décembre',
}

export enum AttendanceSheetStatus {
  ACTIVE = 'ACTIVE',
  INACTIVE = 'INACTIVE', // Correction de l'orthographe
}

export const attendanceSheetStatusLabelsFr: Record<AttendanceSheetStatus, string> = {
  [AttendanceSheetStatus.ACTIVE]: 'Actif',
  [AttendanceSheetStatus.INACTIVE]: 'Inactif',
};

export interface AttendanceSheet {
  id: string;
  number: string;
  year: number;
  month: string | number;
  departmentId: number;
  departmentName: string | null;
  circuitId: string;
  circuitName: string | null;
  status: AttendanceSheetStatus;
  passengers: PassengerDto[];
  presenceDaysCount: number;
  absenceDaysCount: number;
  schoolDaysCount: number;
  transportArCount: number;
  createdAt: string;
  updatedAt: string;
  isActive: boolean;
  calendar: any;
  establishment: string | null;
}
export interface AttendanceSheetWithRouteSheets extends AttendanceSheet {
  routeSheets?: RouteSheetDTO[];
}
export interface AttendanceSheetList {
  content: AttendanceSheet[];
  totalElements: number;
  totalPages: number;
  page: number;
  size: number;
}
export interface CreateAttendanceSheetPayload {
  circuitId: string;
  departmentId: number;
  year: number;
  month: number | string;
}
export interface Circuit {
  id: string;
  titled: string;
}
export interface AttendanceSheetPageFilterParams {
  number?: string;
  circuitName?: string;
  departmentName?: string;
  year?: number;
  month?: string | number | Months;
  page?: number;
  size?: number;
  sortBy?: string;
  sortDirection?: 'asc' | 'desc';
  status?: AttendanceSheetStatus;
}

export interface AttendanceSheetRequest {
  id?: string;
  number: string;
  year: number;
  month: string | number;
  departmentId: number;
  circuitId: string;
  status?: AttendanceSheetStatus;
  presenceDaysCount?: number;
  absenceDaysCount?: number;
  schoolDaysCount?: number;
  transportArCount?: number;
  isActive?: boolean;
}

export interface AttendanceSheetSearchDetails {
  id: string;
  nomCircuit?: string;
  departement?: string;
  mois?: string | Months;
  annee?: string | number;
  nom?: string;
  prenom?: string;
  etablissement?: string;
  telephone?: string;
  joursClasse?: string | number;
  joursPresence?: string | number;
  joursAbsence?: string | number;
  transportAR?: string | number;
  completionDate?: string;
  nomResponsable?: string;
}

export const mapAttendanceSheetToRequest = (sheet: AttendanceSheet): AttendanceSheetRequest => {
  return {
    id: sheet.id,
    number: sheet.number,
    year: sheet.year,
    month: sheet.month,
    departmentId: sheet.departmentId,
    circuitId: sheet.circuitId,
    status: sheet.status,
    presenceDaysCount: sheet.presenceDaysCount,
    absenceDaysCount: sheet.absenceDaysCount,
    schoolDaysCount: sheet.schoolDaysCount,
    transportArCount: sheet.transportArCount,
    isActive: sheet.isActive,
  };
};
export interface PassengerDto {
  id: number;
  firstName: string;
  lastName: string;
  establishmentName?: string;
  departmentName?: string;
}

export interface EstablishmentResponseDTO {
  id: number;
  establishmentName: string;
  fax: string;
  address: {
    street: string;
    zipCode: string;
    city: string;
    country: string;
  };
  establishmentManager: {
    id?: number;
    firstName?: string;
    lastName?: string;
    email?: string;
  };

}

export interface UserResponseDTO {
  id?:number;
  firstName?:string;
  lastName?:string;
}

export interface CalendarDayDTO {
  dayOfMonth: number;
  dayOfWeek: string;
  passengerStatuses: { [passengerId: string]: string };
  weekend: boolean;
  holiday: boolean;
}

export interface AttendanceCalendarDTO {
  days: CalendarDayDTO[];
  passengers: PassengerDto[];
}

export interface RouteSheetDTO {
  id:string;
  attendanceSheetNumber: string;
  circuitName: string;
  departmentName: string;
  establishment: EstablishmentResponseDTO | null;
  month: number;
  year: number;
  driver: UserResponseDTO | null;
  passengers: PassengerDto[];
  generatedAt: string;
  calendar: AttendanceCalendarDTO;
   observation?: string;
}
export interface IAttendanceNotCher {
  departement?: string;
  month?: string;
  year?: string; 
  driver?: string;
  circuit?: string;
  joursAbsence?:any;
  joursPresence?:any;
  etablissement?: string;
  eleveA?: string;
  eleveB?: string;
  eleveC?: string;
  eleveD?: string;
  eleveE?: string;
  observation?: string;
  passengers?: PassengerDto[];
  calendar?: AttendanceCalendarDTO; 
}
export interface IAttendanceSheetCher {
  id: string;
  circuit?: string;
  departement?: string;
  month?: string;
  year?: string;  
  nom?: string;
  prenom?: string;
  establishment?: {
    id: number;
    establishmentName: string;
    fax: string;
    address: {
      street: string;
      zipCode: string;
      city: string;
      country: string;
    };
  };
  etablissement?: string;
  adresse?: string;
  telephone?: string;
  calendar?: AttendanceCalendarDTO; 
  joursClasse?: string;
  joursPresence?: string;
  joursAbsence?: string;
  transportAR?: string;
  completionDate?: string;
  nomResponsable?: string;
  passengers?: PassengerDto[];
}
export enum Months {
  Janvier = 1,
  Février = 2,
  Mars = 3,
  Avril = 4,
  Mai = 5,
  Juin = 6,
  Juillet = 7,
  Août = 8,
  Septembre = 9,
  Octobre = 10,
  Novembre = 11,
  Décembre = 12
}

// Fonction utilitaire pour convertir le numéro en nom de mois
export const monthNumberToName = (monthNumber: number): string => {
  return Object.entries(Months).find(([_, value]) => value === monthNumber)?.[0] || '';
};

// Fonction inverse si nécessaire
export const monthNameToNumber = (monthName: string): number => {
  return Months[monthName as keyof typeof Months];
};

export interface BatchUpdatePayload {
  attendanceSheetId: number;
  changes: Array<{
    passengerId: number;
    date: string;
    concernedTrips: ConcernedTrip[];
  }>;
  observation?: string;
}
export interface PassengerAttendanceDTO {
  passengerId: number;
  passengerName: string;
  concernedTrips: string[];
}
export interface DayDetailDTO {
  date: string; 
  dayOfWeek: string;
  weekend: boolean; 
  passengerAttendances: PassengerAttendanceDTO[]; 
}


export interface AttendanceStatisticsDTO {
  schoolDaysCount: number;
  presenceDaysCount: number;
  absenceDaysCount: number;
  transportArCount: number;
}
export interface AttendanceSheetDetail {
  number: string;
  departmentName: string;
  circuitName: string;
  month: number;
  year: number;
  establishment: EstablishmentResponseDTO | null;
  driver: UserResponseDTO | null;
  passengers: PassengerDto[];
  days: DayDetailDTO[];
  statistics: AttendanceStatisticsDTO;
}