package com.marketingconfort.adanev.vsn.policy;
import org.yaml.snakeyaml.Yaml;

import java.io.IOException;
import java.io.InputStream;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.StandardOpenOption;
import java.util.*;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.regex.Pattern;
import java.util.stream.Collectors;



public class Policy {
    private static final Logger LOGGER = Logger.getLogger(Policy.class.getName());

    private final InputStream yamlStream;
    private final Path javaOutput;
    private final Path tsOutput;

    public Policy() throws Exception {
        String pkg = getClass().getPackageName();
        Path javaDir = Paths.get("src/main/java").resolve(pkg.replace('.', '/'));
        Path tsDir = Paths.get("src/main/resources/static");
        Files.createDirectories(javaDir);

        Files.createDirectories(tsDir);
        this.javaOutput = javaDir.resolve("../user/enums/Permission.java");
        this.tsOutput = tsDir.resolve("permissions.gen.ts");

        this.yamlStream = Optional.ofNullable(
                getClass().getClassLoader().getResourceAsStream("policy/permissions.yml")
        ).orElseThrow(() -> new IllegalStateException("permissions.yml not found on classpath"));
    }

    public static void main(String[] args) {
        try {
            new Policy().run();
            LOGGER.log(Level.INFO, "Permission generator finished.");
        } catch (Exception e) {
            LOGGER.log(Level.SEVERE, "Exception during permission generation", e);
            System.exit(1);
        }
    }

    public void run() throws IOException {
        Config cfg = new Yaml().loadAs(yamlStream, Config.class);
        validateConfig(cfg);

        List<PermissionEntry> entries = new ArrayList<>();
        flattenTree("", cfg.permissions, entries);
        ensureUnique(entries.stream().map(e -> e.path).collect(Collectors.toList()));

        String javaSrc = renderJava(entries);
        Files.writeString(javaOutput, javaSrc, StandardCharsets.UTF_8,
                StandardOpenOption.CREATE, StandardOpenOption.TRUNCATE_EXISTING);

        String tsSrc = renderTs(entries, cfg.permissions);
        Files.writeString(tsOutput, tsSrc, StandardCharsets.UTF_8,
                StandardOpenOption.CREATE, StandardOpenOption.TRUNCATE_EXISTING);
    }

    private void validateConfig(Config cfg) {
        Set<String> modules = new HashSet<>();
        collectModules("", cfg.permissions, modules);
        Set<String> allPerms = new HashSet<>(modules);
        modules.forEach(m -> {
            PermissionNode node = getNode(cfg.permissions, m);
            if (node.hierarchy != null) {
                node.hierarchy.forEach(op -> allPerms.add(m + ":" + op));
            }
        });

        Pattern simple = Pattern.compile("^[A-Z][A-Z0-9_]*$");
        Pattern ref = Pattern.compile("^([A-Z0-9_]+:)?[A-Z0-9_]+$");

        modules.forEach(m -> {
            PermissionNode node = getNode(cfg.permissions, m);
            if (node.hierarchy != null) node.hierarchy.forEach(op -> {
                if (!simple.matcher(op).matches()) {
                    throw new IllegalArgumentException("Bad hierarchy token '" + op + "' in " + m);
                }
                String full = m + ":" + op;
                if (!allPerms.contains(full)) {
                    throw new IllegalArgumentException("Undefined hierarchy perm '" + full + "'");
                }
            });
            if (node.implies != null) node.implies.forEach(refPerm -> {
                if (!ref.matcher(refPerm).matches()) {
                    throw new IllegalArgumentException("Bad implies token '" + refPerm + "' in " + m);
                }
                String full = refPerm.contains(":") ? refPerm : (m + ":" + refPerm);
                if (!allPerms.contains(full)) {
                    throw new IllegalArgumentException("Undefined implies perm '" + full + "'");
                }
            });
        });
    }

    private void collectModules(String prefix,
                                Map<String, PermissionNode> tree,
                                Set<String> out) {
        tree.forEach((key, node) -> {
            String path = prefix.isEmpty() ? key : (prefix + ":" + key);
            out.add(path);
            if (node.children != null) {
                collectModules(path, node.children, out);
            }
        });
    }

    private PermissionNode getNode(Map<String, PermissionNode> tree, String path) {
        String[] parts = path.split(":");
        Map<String, PermissionNode> cur = tree;
        PermissionNode node = null;
        for (String p : parts) {
            node = cur.get(p);
            cur = (node.children == null ? Collections.emptyMap() : node.children);
        }
        return node;
    }

    private void flattenTree(String prefix,
                             Map<String, PermissionNode> tree,
                             List<PermissionEntry> out) {
        tree.forEach((key, node) -> {
            String path = prefix.isEmpty() ? key : (prefix + ":" + key);
            out.add(new PermissionEntry(path, node.desc));

            if (node.hierarchy != null) {
                for (String op : node.hierarchy) {
                    String childPath = path + ":" + op;
                    String childDesc = Optional.ofNullable(node.children)
                            .map(ch -> ch.get(op))
                            .map(cn -> cn.desc)
                            .orElse(null);
                    out.add(new PermissionEntry(childPath, childDesc));
                }
            }
            if (node.children != null) {
                flattenTree(path, node.children, out);
            }
        });
    }

    private void ensureUnique(List<String> list) {
        Set<String> seen = new HashSet<>();
        for (String s : list) {
            if (!seen.add(s)) {
                throw new IllegalStateException("Duplicate permission: " + s);
            }
        }
    }

    private String renderJava(List<PermissionEntry> entries) {
        String pkg = getClass().getPackageName().replace(".policy", ".user.enums");
        StringBuilder sb = new StringBuilder()
                .append("// AUTO-GENERATED; DO NOT EDIT\n")
                .append("package ").append(pkg).append(";\n\n")
                .append("import java.util.*;\n\n")
                .append("public enum Permission {\n");

        entries.stream()
                .sorted(Comparator.comparing(e -> e.path))
                .forEach(e -> {
                    if (e.desc != null) sb.append("    /** ").append(e.desc).append(" */\n");
                    sb.append("    ")
                            .append(e.path.replace(':', '_'))
                            .append("(\"").append(e.path).append("\"),\n");
                });
        int comma = sb.lastIndexOf(",");
        sb.replace(comma, comma + 1, ";\n\n");

        sb.append("    private final String path;\n")
                .append("    Permission(String path) { this.path = path; }\n\n")
                .append("    public String path() { return path; }\n\n");

        sb.append("    private static final Map<Permission, List<Permission>> hierarchy = new HashMap<>();\n")
                .append("    private static final Map<Permission, List<Permission>> implications = new HashMap<>();\n\n")
                .append("    static {\n")
                .append("        for (Permission p: values()) { hierarchy.put(p, new ArrayList<>()); implications.put(p, new ArrayList<>()); }\n");
        entries.stream()
                .filter(e -> e.path.contains(":"))
                .forEach(e -> {
                    String[] parts = e.path.split(":");
                    if (parts.length == 3) {
                        String parent = parts[0] + ":" + parts[1];
                        sb.append("        hierarchy.get(")
                                .append(parent.replace(':', '_')).append(")")
                                .append(".add(")
                                .append(e.path.replace(':', '_')).append(");\n");
                    }
                });
        sb.append("    }\n\n");

        sb.append("    public static Set<Permission> getAllDerivedPermissions(Collection<Permission> perms) {\n")
                .append("        Set<Permission> result = new LinkedHashSet<>();\n")
                .append("        perms.forEach(p -> traverse(p, result));\n")
                .append("        return result;\n")
                .append("    }\n\n")
                .append("    private static void traverse(Permission p, Set<Permission> seen) {\n")
                .append("        if (!seen.add(p)) return;\n")
                .append("        hierarchy.get(p).forEach(c -> traverse(c, seen));\n")
                .append("        implications.get(p).forEach(c -> traverse(c, seen));\n")
                .append("    }\n")
                .append("}\n");
        return sb.toString();
    }

    private String renderTs(
            List<PermissionEntry> entries,
            Map<String, PermissionNode> tree
    ) {
        List<String> allEnumKeys = entries.stream()
                .map(e -> e.path.replace(':', '_'))
                .sorted()
                .toList();

        Map<String, List<String>> hierarchyMap = new LinkedHashMap<>();
        tree.forEach((res, node) -> {
            if (node.children != null) {
                node.children.forEach((action, def) -> {
                    String parent = res + ":" + action;
                    if (def.hierarchy != null && !def.hierarchy.isEmpty()) {
                        List<String> children = def.hierarchy.stream()
                                .map(op -> op.contains(":") ? op : (parent + ":" + op))
                                .map(p -> p.replace(':', '_'))
                                .toList();
                        hierarchyMap.put(parent.replace(':', '_'), children);
                    }
                });
            }
        });

        Map<String, List<String>> impliesMap = new LinkedHashMap<>();
        tree.forEach((res, node) -> {
            if (node.children != null) {
                node.children.forEach((action, def) -> {
                    String parent = res + ":" + action;
                    if (def.implies != null && !def.implies.isEmpty()) {
                        List<String> children = def.implies.stream()
                                .map(ref -> ref.contains(":") ? ref : (parent + ":" + ref))
                                .map(p -> p.replace(':', '_'))
                                .toList();
                        impliesMap.put(parent.replace(':', '_'), children);
                    }
                });
            }
        });

        StringBuilder sb = new StringBuilder()
                .append("// AUTO-GENERATED; DO NOT EDIT\n")
                .append("export enum Permission {\n");
        entries.stream()
                .sorted(Comparator.comparing(e -> e.path))
                .forEach(e -> {
                    if (e.desc != null && !e.desc.isBlank()) {
                        sb.append("  /** ").append(e.desc).append(" */\n");
                    }
                    String key = e.path.replace(':', '_');
                    sb.append("  ").append(key)
                            .append(" = \"").append(e.path).append("\",\n");
                });
        sb.append("}\n\n");

        sb.append("export const PermissionHierarchy: Record<Permission, Permission[]> = {\n");
        for (String key : allEnumKeys) {
            List<String> kids = hierarchyMap.getOrDefault(key, List.of());
            sb.append("  [Permission.").append(key).append("]: [");
            sb.append(kids.stream().map(k -> "Permission." + k).collect(Collectors.joining(", ")));
            sb.append("],\n");
        }
        sb.append("};\n\n");

        sb.append("export const PermissionImplications: Record<Permission, Permission[]> = {\n");
        for (String key : allEnumKeys) {
            List<String> kids = impliesMap.getOrDefault(key, List.of());
            sb.append("  [Permission.").append(key).append("]: [");
            sb.append(kids.stream().map(k -> "Permission." + k).collect(Collectors.joining(", ")));
            sb.append("],\n");
        }
        sb.append("};\n\n");

        sb.append("/** Checks if a string is a known Permission. */\n")
                .append("export function hasPermission(p: string): p is Permission {\n")
                .append("  return Object.values(Permission).includes(p as Permission);\n")
                .append("}\n\n")
                .append("/** Throws if the string is not a known Permission. */\n")
                .append("export function validatePermission(p: string): asserts p is Permission {\n")
                .append("  if (!hasPermission(p)) {\n")
                .append("    throw new Error(`Invalid permission: ${p}`);\n")
                .append("  }\n")
                .append("}\n\n")
                .append("/** Recursively collects all derived permissions. */\n")
                .append("export function getAllDerivedPermissions(perms: Permission[]): Permission[] {\n")
                .append("  const result = new Set<Permission>();\n")
                .append("  function traverse(p: Permission) {\n")
                .append("    if (result.has(p)) return;\n")
                .append("    result.add(p);\n")
                .append("    (PermissionHierarchy[p] || []).forEach(traverse);\n")
                .append("    (PermissionImplications[p] || []).forEach(traverse);\n")
                .append("  }\n")
                .append("  perms.forEach(traverse);\n")
                .append("  return Array.from(result);\n")
                .append("}\n");

        return sb.toString();
    }

    private record PermissionEntry(String path, String desc) {
    }

    public static class Config {
        public String version;
        public Map<String, PermissionNode> permissions;
    }

    public static class PermissionNode {
        public String desc;
        public List<String> hierarchy;
        public List<String> implies;
        public Map<String, PermissionNode> children;
    }
}