package com.marketingconfort.adanev.vsn.document.controller;

import com.marketingconfort.adanev.vsn.document.constants.MessageConstants;
import com.marketingconfort.adanev.vsn.document.constants.paths.Paths;
import com.marketingconfort.adanev.vsn.document.dtos.DocumentDTO;
import com.marketingconfort.adanev.vsn.document.dtos.DocumentStatisticsDTO;
import com.marketingconfort.adanev.vsn.document.dtos.requests.AddDocumentRequest;
import com.marketingconfort.adanev.vsn.document.dtos.requests.BulkDocumentRequest;
import com.marketingconfort.adanev.vsn.document.dtos.requests.BulkMoveRequest;
import com.marketingconfort.adanev.vsn.document.enums.ContentType;
import com.marketingconfort.adanev.vsn.document.enums.DocumentType;
import com.marketingconfort.adanev.vsn.document.services.DocumentService;
import com.marketingconfort.starter.core.exceptions.FunctionalException;
import com.marketingconfort.starter.core.exceptions.S3FunctionalException;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.core.io.InputStreamResource;
import org.springframework.data.domain.Page;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.*;
import org.springframework.web.multipart.MultipartFile;

import java.io.IOException;
import java.util.List;

@RestController
@RequestMapping(Paths.DOCUMENT_BASE)
@RequiredArgsConstructor
@Slf4j
public class DocumentController {

    private final DocumentService documentService;

    @PostMapping(Paths.DOCUMENT_UPLOAD)
    public ResponseEntity<String> uploadDocument(
            @RequestParam("file") MultipartFile file,
            AddDocumentRequest request) throws FunctionalException, IOException, S3FunctionalException {

        if (request.getFolderId() != null) {
            documentService.importDocument(file, request);
        } else {
            documentService.importDocumentToRoot(file, request);
        }
        return ResponseEntity.ok(MessageConstants.SUCCESSFUL_UPLOAD);
    }

    @GetMapping(Paths.USER_DOCUMENTS)
    public ResponseEntity<List<DocumentDTO>> getDocumentsByUser(@RequestParam Long ownerId) throws FunctionalException {
        return ResponseEntity.ok(documentService.getDocumentsByOwnerId(ownerId));
    }

    @GetMapping(Paths.DOCUMENT_DETAILS)
    public ResponseEntity<DocumentDTO> getDocumentDetails(@RequestParam Long documentId, @RequestParam Long ownerId) throws FunctionalException {
        return ResponseEntity.ok(documentService.getDocumentByIdAndOwner(documentId, ownerId));
    }

    @GetMapping(Paths.ROOT_DOCUMENTS)
    public ResponseEntity<List<DocumentDTO>> getRootDocuments(@RequestParam Long ownerId) {
        return ResponseEntity.ok(documentService.getRootDocumentsByOwnerId(ownerId));
    }

    @GetMapping(Paths.DOCUMENT_STATS)
    public ResponseEntity<List<DocumentStatisticsDTO>> getDocumentStats(@RequestParam Long ownerId) {
        List<DocumentStatisticsDTO> stats = documentService.getDocumentStatistics(ownerId);
        return ResponseEntity.ok(stats);
    }

    @GetMapping(Paths.GET_DOWNLOAD_URL)
    public ResponseEntity<String> getDocumentDownloadLink(@RequestParam Long documentId,@RequestParam Long ownerId) throws FunctionalException {
        String downloadUrl = documentService.getDocumentDownloadLink(documentId, ownerId);
        return ResponseEntity.ok(downloadUrl);
    }

    @GetMapping(Paths.DOWNLOAD_ELEMENT)
    public ResponseEntity<InputStreamResource> downloadDocument(
            @RequestParam Long documentId,
            @RequestParam Long ownerId) throws FunctionalException {
        return documentService.downloadDocument(documentId, ownerId);
    }

    @GetMapping(Paths.SEARCH_DOCUMENTS)
    public ResponseEntity<List<DocumentDTO>> searchDocuments(
            @RequestParam Long ownerId,
            @RequestParam(required = false) String keyword,
            @RequestParam(required = false, defaultValue = "name") String sortBy,
            @RequestParam(required = false, defaultValue = "true") boolean asc) {

        List<DocumentDTO> results = documentService.searchAndSortDocuments(ownerId, keyword, sortBy, asc);
        return ResponseEntity.ok(results);
    }

    @GetMapping(Paths.SEARCH_DOCUMENTS_ADVANCED)
    public ResponseEntity<Page<DocumentDTO>> advancedSearchDocuments(
            @RequestParam Long ownerId,
            @RequestParam(required = false) String keyword,
            @RequestParam(required = false) ContentType contentType,
            @RequestParam(required = false) DocumentType documentType,
            @RequestParam(defaultValue = "name") String sortBy,
            @RequestParam(defaultValue = "true") boolean asc,
            @RequestParam(defaultValue = "0") int page,
            @RequestParam(defaultValue = "10") int size
    ) {
        Page<DocumentDTO> result = documentService.advancedSearchDocuments(
                ownerId, keyword, contentType, documentType, sortBy, asc, page, size
        );
        return ResponseEntity.ok(result);
    }

    @DeleteMapping(Paths.DELETE_DOCUMENT)
    public ResponseEntity<Void> deleteDocument(
            @RequestParam Long documentId,
            @RequestParam Long ownerId) throws FunctionalException {
        documentService.deleteDocument(documentId, ownerId);
        return ResponseEntity.ok().build();
    }

    @PostMapping(Paths.BULK_DELETE_DOCUMENTS)
    public ResponseEntity<Void> bulkDeleteDocuments(
            @RequestBody BulkDocumentRequest request
    ) throws FunctionalException {
        documentService.bulkDeleteDocuments(request.getDocumentIds(), request.getOwnerId());
        return ResponseEntity.ok().build();
    }

    @PostMapping(Paths.DOCUMENT_SHARE)
    public ResponseEntity<Void> shareDocument(@RequestParam Long documentId, @RequestParam Long targetUserId) throws S3FunctionalException, FunctionalException {
        documentService.shareDocument(documentId, targetUserId);
        return ResponseEntity.ok().build();
    }

    @PostMapping(Paths.BULK_SHARE_DOCUMENTS)
    public ResponseEntity<Void> bulkShareDocument(@RequestBody BulkDocumentRequest bulkDocumentRequest) throws S3FunctionalException, FunctionalException {
     documentService.bulkShareDocuments(bulkDocumentRequest.getDocumentIds(), bulkDocumentRequest.getOwnerId());
     return ResponseEntity.ok().build();

    }

    @PostMapping(Paths.MARK_DOC_AS_FAVORITE)
    public ResponseEntity<Void> markAsFavorite(
            @PathVariable Long documentId,
            @RequestParam Long ownerId) throws FunctionalException {
        documentService.markAsFavorite(documentId, ownerId);
        return ResponseEntity.ok().build();
    }

    @DeleteMapping(Paths.UNMARK_DOC_AS_FAVORITE)
    public ResponseEntity<Void> unmarkAsFavorite(
            @PathVariable Long documentId,
            @RequestParam Long ownerId) throws FunctionalException {
        documentService.unmarkAsFavorite(documentId, ownerId);
        return ResponseEntity.ok().build();
    }

    @GetMapping(Paths.FAVORITE_DOCS)
    public ResponseEntity<List<DocumentDTO>> getFavoriteDocuments(
            @RequestParam Long ownerId) {
        return ResponseEntity.ok(documentService.getFavoriteDocuments(ownerId));
    }

    @PostMapping(Paths.MOVE_DOCUMENT)
    public ResponseEntity<Void> moveDocument(
            @PathVariable Long documentId,
            @RequestParam(required = false) Long folderId,
            @RequestParam Long ownerId) throws FunctionalException, S3FunctionalException {
        documentService.moveDocument(documentId, folderId, ownerId);
        return ResponseEntity.ok().build();
    }

    @PostMapping(Paths.BULK_MOVE_DOCUMENTS)
    public ResponseEntity<Void> bulkMoveDocuments(
            @RequestBody BulkMoveRequest request) throws FunctionalException, S3FunctionalException {
        documentService.bulkMoveDocuments(request.getDocumentIds(), request.getTargetFolderId(), request.getOwnerId());
        return ResponseEntity.ok().build();
    }


}
