package com.marketingconfort.adanev.vsn.document.services.Implementations;

import com.amazonaws.services.s3.AmazonS3;
import com.amazonaws.services.s3.model.AmazonS3Exception;
import com.amazonaws.services.s3.model.CopyObjectRequest;
import com.marketingconfort.adanev.vsn.document.config.AWSConfig;
import com.marketingconfort.adanev.vsn.document.config.EnvConfigLoader;
import com.marketingconfort.adanev.vsn.document.constants.MessageConstants;
import com.marketingconfort.adanev.vsn.document.dtos.DocumentDTO;
import com.marketingconfort.adanev.vsn.document.dtos.DocumentStatisticsDTO;
import com.marketingconfort.adanev.vsn.document.dtos.requests.AddDocumentRequest;
import com.marketingconfort.adanev.vsn.document.enums.DocumentType;
import com.marketingconfort.adanev.vsn.document.mappers.DocumentMapper;
import com.marketingconfort.adanev.vsn.document.models.Document;
import com.marketingconfort.adanev.vsn.document.enums.ContentType;
import com.marketingconfort.adanev.vsn.document.models.Folder;
import com.marketingconfort.adanev.vsn.document.repositories.DocumentRepository;
import com.marketingconfort.adanev.vsn.document.repositories.FolderRepository;
import com.marketingconfort.adanev.vsn.document.services.DocumentService;
import com.marketingconfort.adanev.vsn.document.services.StorageQuotaService;
import com.marketingconfort.adanev.vsn.document.utils.DocumentUtils;
import com.marketingconfort.adanev.vsn.document.utils.FolderSizeCalculator;
import com.marketingconfort.starter.core.exceptions.FunctionalException;
import com.marketingconfort.starter.core.exceptions.S3FunctionalException;
import com.marketingconfort.starter.core.exceptions.enums.S3FunctionalExceptionType;
import com.marketingconfort.starter.core.services.S3FileService;
import jakarta.persistence.criteria.Predicate;
import lombok.RequiredArgsConstructor;
import org.springframework.core.io.InputStreamResource;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.data.domain.Sort;
import org.springframework.http.HttpHeaders;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.web.multipart.MultipartFile;

import java.io.IOException;
import java.net.URL;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

@Service
@RequiredArgsConstructor
public class DocumentServiceImpl implements DocumentService {

    private final FolderRepository folderRepository;
    private final DocumentRepository documentRepository;
    private final StorageQuotaService storageQuotaService;
    private final DocumentStorageService documentStorageService;
    private final S3FileService s3FileService;
    private static final AmazonS3 amazonS3Client = AWSConfig.getInstance().amazonS3();
    private final DocumentMapper documentMapper;
    private final String bucketName = EnvConfigLoader.getEnvironmentConfig().getAws().getBucketName();

    @Transactional
    @Override
    public void importDocument(MultipartFile file, AddDocumentRequest request) throws FunctionalException, IOException, S3FunctionalException {
        Folder folder = null;
        if (request.getFolderId() != null) {
            folder = folderRepository.findById(request.getFolderId())
                    .orElseThrow(() -> new FunctionalException(MessageConstants.FOLDER_NOT_FOUND));
            if (!isFolderOwnedByUser(folder.getId(), request.getOwnerId())) {
                throw new FunctionalException(MessageConstants.UNAUTHORIZED_ACCESS);
            }
        }

        if (!storageQuotaService.hasAvailableStorage(request.getOwnerId(), file.getSize())) {
            throw new FunctionalException(MessageConstants.STORAGE_QUOTA_EXCEEDED);
        }

        if (!isDocumentNameUnique(file.getOriginalFilename(), request.getFolderId(), request.getOwnerId())) {
            throw new FunctionalException(MessageConstants.DUPLICATE_NAME);
        }

        String s3Key = DocumentUtils.constructS3Key(file, request.getOwnerId(), folder);
        s3FileService.uploadElementToBucket("file", s3Key, file.getOriginalFilename(), file, bucketName);

        String path = DocumentUtils.constructDocumentPath(file.getOriginalFilename(), folder);
        ContentType docType = DocumentUtils.detectContentType(file);
        Document document = new Document();
        document.setName(request.getName() != null ? request.getName() : file.getOriginalFilename());
        document.setPath(path);
        document.setSize(file.getSize());
        document.setContentType(docType);
        document.setS3Key(s3Key + file.getOriginalFilename());
        document.setFolder(folder);
        document.setOwnerId(request.getOwnerId());
        document.setDocumentType(request.getDocumentType());
        if (folder != null) {
            folder.getDocuments().add(document);
            folderRepository.save(folder);
        }

        documentRepository.save(document);
    }

    @Override
    @Transactional
    public void importDocumentToRoot(MultipartFile file,AddDocumentRequest request) throws FunctionalException, IOException, S3FunctionalException {
        importDocument(file, request);
    }

    @Override
    @Transactional
    public List<DocumentDTO> getDocumentsByOwnerId(Long ownerId) {
        List<Document> documents = documentRepository.findByOwnerId(ownerId);
        return documents.stream().map(documentMapper::toDto).collect(Collectors.toList());
    }

    @Transactional
    @Override
    public DocumentDTO getDocumentByIdAndOwner(Long documentId, Long ownerId) throws FunctionalException {
        Document document = documentRepository.findByIdAndOwnerId(documentId, ownerId)
                .orElseThrow(() -> new FunctionalException(MessageConstants.DOCUMENT_NOT_FOUND));
        return documentMapper.toDto(document);
    }

    @Override
    public List<DocumentDTO> getRootDocumentsByOwnerId(Long ownerId) {
        List<Document> documents = documentRepository.findByOwnerIdAndFolderIsNull(ownerId);
        return documents.stream().map(documentMapper::toDto).collect(Collectors.toList());
    }

    @Override
    public List<DocumentStatisticsDTO> getDocumentStatistics(Long ownerId) {
        List<Document> userDocuments = documentRepository.findByOwnerId(ownerId);

        Map<DocumentType, List<Document>> groupedByType = userDocuments.stream()
                .filter(doc -> doc.getDocumentType() != null)
                .collect(Collectors.groupingBy(Document::getDocumentType));

        return groupedByType.entrySet().stream().map(entry -> {
            DocumentType type = entry.getKey();
            List<Document> docs = entry.getValue();
            long totalSize = docs.stream().filter(d -> d.getSize() != null).mapToLong(Document::getSize).sum();
            long count = docs.size();
            long avgSize = count > 0 ? totalSize / count : 0;

            return DocumentStatisticsDTO.builder()
                    .documentType(type)
                    .documentCount(count)
                    .totalSizeBytes(totalSize)
                    .totalSizeFormatted(FolderSizeCalculator.formatBytes(totalSize))
                    .averageSizeFormatted(FolderSizeCalculator.formatBytes(avgSize))
                    .build();
        }).collect(Collectors.toList());
    }

    @Override
    public String getDocumentDownloadLink(Long documentId, Long ownerId) throws FunctionalException {
        Document document = documentRepository.findById(documentId)
                .orElseThrow(() -> new FunctionalException(MessageConstants.DOCUMENT_NOT_FOUND));

        if (document.getOwnerId() != ownerId) {
            throw new FunctionalException(MessageConstants.UNAUTHORIZED_ACCESS);
        }

        return documentStorageService.getFileSharableLink(bucketName, document.getS3Key());
    }

    @Override
    public ResponseEntity<InputStreamResource> downloadDocument(Long documentId, Long ownerId) throws FunctionalException {
        try {
            String presignedUrl = getDocumentDownloadLink(documentId, ownerId); // validates access
            Document document = documentRepository.findById(documentId)
                    .orElseThrow(() -> new FunctionalException(MessageConstants.DOCUMENT_NOT_FOUND));

            String fileName = document.getName();

            HttpHeaders headers = new HttpHeaders();
            headers.add(HttpHeaders.CONTENT_DISPOSITION, "attachment; filename=\"" + fileName + "\"");

            return ResponseEntity.ok()
                    .headers(headers)
                    .contentType(MediaType.APPLICATION_OCTET_STREAM)
                    .body(new InputStreamResource(new URL(presignedUrl).openStream()));

        } catch (IOException e) {
            throw new FunctionalException(MessageConstants.FILE_DOWNLOAD_FAILED);
        }
    }

    @Override
    public List<DocumentDTO> searchAndSortDocuments(Long ownerId, String keyword, String sortBy, boolean asc) {
        Sort sort = Sort.by(asc ? Sort.Direction.ASC : Sort.Direction.DESC, resolveSortField(sortBy));
        List<Document> results = documentRepository.searchDocuments(ownerId, keyword, sort);
        return results.stream().map(documentMapper::toDto).collect(Collectors.toList());
    }

    public Page<DocumentDTO> advancedSearchDocuments(
            Long ownerId,
            String keyword,
            ContentType contentType,
            DocumentType documentType,
            String sortBy,
            boolean asc,
            int page,
            int size) {

        Pageable pageable = PageRequest.of(
                page,
                size,
                Sort.by(asc ? Sort.Direction.ASC : Sort.Direction.DESC, resolveSortField(sortBy))
        );

        Page<Document> documentPage = documentRepository.findAll((root, query, cb) -> {
            List<Predicate> predicates = new ArrayList<>();

            predicates.add(cb.equal(root.get("ownerId"), ownerId));

            if (keyword != null && !keyword.trim().isEmpty()) {
                predicates.add(cb.like(cb.lower(root.get("name")), "%" + keyword.trim().toLowerCase() + "%"));
            }

            if (contentType != null) {
                predicates.add(cb.equal(root.get("contentType"), contentType));
            }

            if (documentType != null) {
                predicates.add(cb.equal(root.get("documentType"), documentType));
            }

            return cb.and(predicates.toArray(new Predicate[0]));
        }, pageable);

        return documentPage.map(documentMapper::toDto);
    }
    @Override
    @Transactional
    public void deleteDocument(Long documentId, Long ownerId) throws FunctionalException {
        Document document = documentRepository.findById(documentId)
                .orElseThrow(() -> new FunctionalException(MessageConstants.DOCUMENT_NOT_FOUND));

        if (document.getOwnerId() != ownerId) {
            throw new FunctionalException(MessageConstants.UNAUTHORIZED_ACCESS);
        }

        // Remove document from folder if present
        Folder folder = document.getFolder();
        if (folder != null) {
            folder.getDocuments().remove(document);
            document.setFolder(null);
        }

        // Delete from S3
        try {
            s3FileService.deleteSingleFileInBucket(bucketName, document.getS3Key());
        } catch (S3FunctionalException e) {
            throw new FunctionalException(MessageConstants.S3_ERROR + e.getMessage());
        }

        // Delete from DB
        documentRepository.delete(document);
    }

    @Override
    @Transactional
    public void bulkDeleteDocuments(List<Long> documentIds, Long ownerId) throws FunctionalException {
        List<Document> documents = documentRepository.findAllById(documentIds);

        // Authorization check
        for (Document document : documents) {
            if (!(document.getOwnerId() == ownerId)) {
                throw new FunctionalException(MessageConstants.UNAUTHORIZED_ACCESS);
            }
        }

        // Remove document references from folders and collect S3 keys
        List<String> s3KeysToDelete = new ArrayList<>();
        for (Document document : documents) {
            Folder folder = document.getFolder();
            if (folder != null) {
                folder.getDocuments().remove(document);
                document.setFolder(null);
            }

            s3KeysToDelete.add(document.getS3Key());
        }

        // Delete from S3
        try {
            s3FileService.deleteFilesInBucket(bucketName, s3KeysToDelete);
        } catch (S3FunctionalException e) {
            throw new FunctionalException(MessageConstants.S3_ERROR + e.getMessage());
        }

        // Delete from DB
        documentRepository.deleteAll(documents);
    }

    @Override
    @Transactional
    public void bulkShareDocuments(List<Long> documentIds, Long targetUserId) throws FunctionalException, S3FunctionalException {
        // Vérification ou création du dossier "sharedWithMe" pour l'utilisateur cible
        Folder sharedFolder = getOrCreateSharedWithMe(targetUserId);

        for (Long documentId : documentIds) {
            Document originalDoc = documentRepository.findById(documentId)
                    .orElseThrow(() -> new FunctionalException(MessageConstants.DOCUMENT_NOT_FOUND));

            // Génère le nouveau chemin et S3 key
            String newPath = DocumentUtils.constructDocumentPath(originalDoc.getName(), sharedFolder);
            String newS3Key = "document_service_folder/" + targetUserId + "/sharedWithMe/" + originalDoc.getName();

            // Copie le fichier sur S3
            amazonS3Client.copyObject(bucketName, originalDoc.getS3Key(), bucketName, newS3Key);

            // Création de la copie du document
            Document sharedDoc = new Document();
            sharedDoc.setName(originalDoc.getName());
            sharedDoc.setPath(newPath);
            sharedDoc.setSize(originalDoc.getSize());
            sharedDoc.setContentType(originalDoc.getContentType());
            sharedDoc.setS3Key(newS3Key);
            sharedDoc.setOwnerId(targetUserId);
            sharedDoc.setDocumentType(originalDoc.getDocumentType());
            sharedDoc.setFolder(sharedFolder);

            documentRepository.save(sharedDoc);
        }
    }
    @Override
    public void shareDocument(Long documentId, Long targetUserId) throws FunctionalException, S3FunctionalException {
        Document originalDoc = documentRepository.findById(documentId)
                .orElseThrow(() -> new FunctionalException(MessageConstants.DOCUMENT_NOT_FOUND));

        // Check or create sharedWithMe folder for target user
        Folder sharedFolder = getOrCreateSharedWithMe(targetUserId);

        String newPath = DocumentUtils.constructDocumentPath(originalDoc.getName(), sharedFolder);  // example : /sharedWithMe/doc.pdf
        String newS3Key = "document_service_folder/" + targetUserId + "/sharedWithMe/" + originalDoc.getName();

        // share a copy on S3
        amazonS3Client.copyObject(bucketName, originalDoc.getS3Key(), bucketName, newS3Key);
        // Create a copy  Document for the target user
        Document sharedDoc = new Document();
        sharedDoc.setName(originalDoc.getName());
        sharedDoc.setPath(newPath);
        sharedDoc.setSize(originalDoc.getSize());
        sharedDoc.setContentType(originalDoc.getContentType());
        sharedDoc.setS3Key(newS3Key);
        sharedDoc.setOwnerId(targetUserId);
        sharedDoc.setDocumentType(originalDoc.getDocumentType());
        sharedDoc.setFolder(sharedFolder);

        documentRepository.save(sharedDoc);
    }

    @Override
    @Transactional
    public void markAsFavorite(Long documentId, Long ownerId) throws FunctionalException {
        Document doc = documentRepository.findById(documentId)
                .orElseThrow(() -> new FunctionalException(MessageConstants.DOCUMENT_NOT_FOUND));

        if (doc.getOwnerId() != ownerId) {
            throw new FunctionalException(MessageConstants.UNAUTHORIZED_ACCESS);
        }

        doc.setFavorite(true);
        documentRepository.save(doc);
    }

    @Override
    @Transactional
    public void unmarkAsFavorite(Long documentId, Long ownerId) throws FunctionalException {
        Document doc = documentRepository.findById(documentId)
                .orElseThrow(() -> new FunctionalException(MessageConstants.DOCUMENT_NOT_FOUND));

        if (doc.getOwnerId() != ownerId) {
            throw new FunctionalException(MessageConstants.UNAUTHORIZED_ACCESS);
        }

        doc.setFavorite(false);
        documentRepository.save(doc);
    }

    @Override
    @Transactional(readOnly = true)
    public List<DocumentDTO> getFavoriteDocuments(Long ownerId) {
        List<Document> docs = documentRepository.findByOwnerIdAndFavoriteTrue(ownerId);
        return docs.stream().map(documentMapper::toDto).collect(Collectors.toList());
    }

    @Override
    @Transactional
    public void moveDocument(Long documentId, Long targetFolderId, Long ownerId) throws FunctionalException, S3FunctionalException {
        Document doc = documentRepository.findById(documentId)
                .orElseThrow(() -> new FunctionalException(MessageConstants.DOCUMENT_NOT_FOUND));

        if (doc.getOwnerId() != ownerId) {
            throw new FunctionalException(MessageConstants.UNAUTHORIZED_ACCESS);
        }

        // Remove from current folder's document list if it exists
        Folder currentFolder = doc.getFolder();
        if (currentFolder != null && currentFolder.getDocuments() != null) {
            currentFolder.getDocuments().remove(doc);
            folderRepository.save(currentFolder);
        }

        // Fetch new folder (target)
        Folder newFolder = null;
        if (targetFolderId != null) {
            newFolder = folderRepository.findById(targetFolderId)
                    .orElseThrow(() -> new FunctionalException(MessageConstants.FOLDER_NOT_FOUND));

            if (!newFolder.getOwnerId().equals(ownerId)) {
                throw new FunctionalException(MessageConstants.UNAUTHORIZED_ACCESS);
            }
        }
        String newPath = DocumentUtils.constructDocumentPath(doc.getName(), newFolder);
        String newS3Key = "document_service_folder/" + doc.getOwnerId() + "/" + doc.getName();

        amazonS3Client.copyObject(bucketName, doc.getS3Key(), bucketName, newS3Key);
        s3FileService.deleteSingleFileInBucket(bucketName, doc.getS3Key());
        // Update document fields
        doc.setPath(newPath);
        doc.setS3Key(newS3Key);
        doc.setFolder(newFolder);

        documentRepository.save(doc);
    }

    @Override
    @Transactional
    public void bulkMoveDocuments(List<Long> documentIds, Long targetFolderId, Long ownerId) throws FunctionalException, S3FunctionalException {
        Folder newFolder = null;

        if (targetFolderId != null) {
            newFolder = folderRepository.findById(targetFolderId)
                    .orElseThrow(() -> new FunctionalException(MessageConstants.FOLDER_NOT_FOUND));

            if (!newFolder.getOwnerId().equals(ownerId)) {
                throw new FunctionalException(MessageConstants.UNAUTHORIZED_ACCESS);
            }
        }

        for (Long documentId : documentIds) {
            Document doc = documentRepository.findById(documentId)
                    .orElseThrow(() -> new FunctionalException(MessageConstants.DOCUMENT_NOT_FOUND));

            if (doc.getOwnerId() != ownerId) {
                throw new FunctionalException(MessageConstants.UNAUTHORIZED_ACCESS);
            }

            // Remove from current folder if exists
            Folder currentFolder = doc.getFolder();
            if (currentFolder != null && currentFolder.getDocuments() != null) {
                currentFolder.getDocuments().remove(doc);
                folderRepository.save(currentFolder);
            }

            // Construct paths
            String oldS3Key = doc.getS3Key();
            String newPath = DocumentUtils.constructDocumentPath(doc.getName(), newFolder);
            String newS3Key = DocumentUtils.constructS3KeyForMove(doc, ownerId, newFolder);

            // Move in S3
            s3FileService.renameOrMoveObject(oldS3Key, newS3Key);

            // Update document
            doc.setPath(newPath);
            doc.setS3Key(newS3Key);
            doc.setFolder(newFolder);

            documentRepository.save(doc);
        }
    }

    private Folder getOrCreateSharedWithMe(Long targetUserId) {
        return folderRepository.findByOwnerIdAndName(targetUserId, "sharedWithMe")
                .orElseGet(() -> {
                    Folder f = new Folder();
                    f.setName("sharedWithMe");
                    f.setOwnerId(targetUserId);
                    f.setFavorite(false);
                    f.setParent(null);
                    return folderRepository.save(f);
                });
    }
    private String resolveSortField(String sortBy) {
        return switch (sortBy != null ? sortBy.toLowerCase() : "") {
            case "name" -> "name";
            case "contenttype" -> "contentType";
            case "documenttype" -> "documentType";
            case "foldername" -> "folder.name";
            default -> "name"; // default fallback
        };
    }
    private boolean isDocumentNameUnique(String name, Long folderId, Long ownerId) {
        if (folderId == null) {
            // Root space: check documents for this user with no folder
            return !documentRepository.existsByNameAndFolderIsNullAndOwnerId(name, ownerId);
        } else {
            return !documentRepository.existsByNameAndFolderIdAndOwnerId(name, folderId, ownerId);
        }
    }
    private boolean isFolderOwnedByUser(Long folderId, Long ownerId) {
        return folderRepository.findById(folderId)
                .map(folder -> folder.getOwnerId() != null && folder.getOwnerId().equals(ownerId))
                .orElse(false);
    }


}

