package com.marketingconfort.adanev.vsn.document.controller;

import com.marketingconfort.adanev.vsn.document.constants.MessageConstants;
import com.marketingconfort.adanev.vsn.document.constants.paths.Paths;
import com.marketingconfort.adanev.vsn.document.dtos.DocumentDTO;
import com.marketingconfort.adanev.vsn.document.dtos.FolderDTO;
import com.marketingconfort.adanev.vsn.document.dtos.requests.BulkShareRequest;
import com.marketingconfort.adanev.vsn.document.dtos.requests.NewFolderRequest;
import com.marketingconfort.adanev.vsn.document.dtos.requests.ShareFolderRequest;
import com.marketingconfort.adanev.vsn.document.dtos.response.FolderDetailsResponse;
import com.marketingconfort.adanev.vsn.document.dtos.response.FolderSizeResponseDTO;
import com.marketingconfort.adanev.vsn.document.services.FolderService;
import com.marketingconfort.starter.core.exceptions.FunctionalException;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.data.domain.Page;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.security.access.prepost.PreAuthorize;
import org.springframework.web.bind.annotation.*;

import java.io.IOException;
import java.util.List;

@RestController
@RequestMapping(Paths.FOLDER_BASE)
@RequiredArgsConstructor
@Slf4j
public class FolderController {

    private final FolderService folderService;

    @PreAuthorize("@authorizationService.isTokenSubjectOwner(#requestAuthorization, #ownerUuid)")
    @GetMapping(Paths.ROOT_FOLDERS)
    public ResponseEntity<List<FolderDTO>> getRootFolders(@PathVariable String ownerUuid) {
        return ResponseEntity.ok(folderService.listRootFolders(ownerUuid));
    }

    @PreAuthorize("@authorizationService.isTokenSubjectOwner(#requestAuthorization, #ownerUuid)")
    @GetMapping(Paths.SUB_FOLDERS)
    public ResponseEntity<List<FolderDTO>> getSubfolders(@PathVariable Long parentId, @RequestParam String ownerUuid) {
        return ResponseEntity.ok(folderService.listSubFolders(parentId, ownerUuid));
    }

    @PreAuthorize("@authorizationService.isTokenSubjectOwner(#requestAuthorization, #ownerUuid)")
    @GetMapping(Paths.FOLDER_DOCS)
    public ResponseEntity<List<DocumentDTO>> getDocuments(@PathVariable Long parentId, @RequestParam String ownerUuid) {
        return ResponseEntity.ok(folderService.listDocumentsInFolder(parentId, ownerUuid));
    }

    @PostMapping(Paths.ADD_FOLDER)
    public ResponseEntity<FolderDTO> createFolder(@RequestBody NewFolderRequest request) throws FunctionalException {
        log.info("Creating new folder: {}", request.getName());
        FolderDTO createdFolder = folderService.createFolder(request);
        log.info("Created new folder: {}", createdFolder.getName());
        return ResponseEntity.status(HttpStatus.CREATED).body(createdFolder);
    }

    @PreAuthorize("@authorizationService.isTokenSubjectOwner(#requestAuthorization, #ownerUuid)")
    @PutMapping(Paths.RENAME_FOLDER)
    public ResponseEntity<FolderDTO> renameFolder(
            @PathVariable Long folderId,
            @RequestParam String newName,
            @RequestParam String ownerUuid) throws FunctionalException {

        FolderDTO updated = folderService.renameFolder(folderId, newName, ownerUuid);
        return ResponseEntity.ok(updated);
    }

    @PreAuthorize("@authorizationService.isTokenSubjectOwner(#requestAuthorization, #ownerUuid)")
    @DeleteMapping(Paths.DELETE_FOLDER)
    public ResponseEntity<Void> deleteFolder(@PathVariable Long folderId, @RequestParam String ownerUuid) throws FunctionalException {
        folderService.deleteFolderById(folderId, ownerUuid);
        return ResponseEntity.status(HttpStatus.NO_CONTENT).build();
    }

    @PreAuthorize("@authorizationService.isTokenSubjectOwner(#requestAuthorization, #ownerUuid)")
    @PutMapping(Paths.MOVE_FOLDER)
    public ResponseEntity<FolderDTO> moveFolder(
            @PathVariable Long folderId,
            @RequestParam (required = false) Long newParentId,
            @RequestParam String ownerUuid) throws FunctionalException {
        return ResponseEntity.ok(folderService.moveFolder(folderId, newParentId, ownerUuid));
    }

    @GetMapping(Paths.FOLDERS_SEARCH)
    public ResponseEntity<List<FolderDTO>> searchFolders(
            @RequestParam String keyword,
            @RequestParam String ownerUuid) {
        return ResponseEntity.ok(folderService.searchFolders(keyword, ownerUuid));
    }

    @PreAuthorize("@authorizationService.isTokenSubjectOwner(#requestAuthorization, #ownerUuid)")
    @GetMapping(Paths.FOLDER_DETAILS)
    public ResponseEntity<FolderDetailsResponse> getFolderDetails(@PathVariable Long folderId) throws FunctionalException {
        FolderDetailsResponse details = folderService.getFolderDetails(folderId);
        return ResponseEntity.ok(details);
    }
    @GetMapping(Paths.FOLDER_ADVANCED_SEARCH)
    public ResponseEntity<Page<FolderDTO>> advancedSearchFolders(
            @RequestParam String keyword,
            @RequestParam String ownerUuid,
            @RequestParam(defaultValue = "0") int page,
            @RequestParam(defaultValue = "10") int size,
            @RequestParam(defaultValue = "name") String sortBy,
            @RequestParam(defaultValue = "ASC") String sortDirection) throws FunctionalException {
        return ResponseEntity.ok(folderService.advancedSearchFolders(keyword, ownerUuid, page, size, sortBy, sortDirection));
    }

    @PreAuthorize("@authorizationService.isTokenSubjectOwner(#requestAuthorization, #ownerUuid)")
    @PostMapping(Paths.FOLDER_SHARE)
    public ResponseEntity<Void> shareFolder(@RequestBody ShareFolderRequest request) throws FunctionalException {
        folderService.shareFolderWithUsers(request);
        return ResponseEntity.ok().build();
    }

    @PreAuthorize("@authorizationService.isTokenSubjectOwner(#requestAuthorization, #ownerUuid)")
    @DeleteMapping(Paths.FOLDER_CANCEL_SHARE)
    public ResponseEntity<Void> unshareFolderWithUser( @PathVariable Long folderId,@PathVariable String userUuid) throws FunctionalException {
        folderService.unshareFolderWithUser(folderId, userUuid);
        return ResponseEntity.noContent().build();
    }

    @PreAuthorize("@authorizationService.isTokenSubjectOwner(#requestAuthorization, #ownerUuid)")
    @GetMapping(Paths.FOLDER_ZIP)
    public ResponseEntity<byte[]> zipFolderAndReturn(@PathVariable Long folderId) throws IOException, FunctionalException {
        byte[] zipData = folderService.zipFolderAndReturn(folderId);

        return ResponseEntity
                .ok()
                .header("Content-Type", "application/zip")
                .header("Content-Disposition", "inline; filename=\"folder_" + folderId + ".zip\"")
                .body(zipData);
    }

    @PreAuthorize("@authorizationService.isTokenSubjectOwner(#requestAuthorization, #ownerUuid)")
    @GetMapping(Paths.DOWNLOAD_ELEMENT)
    public ResponseEntity<byte[]> download(
            @RequestParam(required = false) Long folderId
    ) throws FunctionalException, IOException {
        return folderService.downloadZippedFolder(folderId);
    }

    @PreAuthorize("@authorizationService.isTokenSubjectOwner(#requestAuthorization, #ownerUuid)")
    @GetMapping(Paths.FOLDER_SIZE)
    public ResponseEntity<FolderSizeResponseDTO> getFolderSize(@PathVariable Long folderId) throws FunctionalException {
        FolderSizeResponseDTO response = folderService.calculateFolderSize(folderId);
        return ResponseEntity.ok(response);
    }

    @PreAuthorize("@authorizationService.isTokenSubjectOwner(#requestAuthorization, #ownerUuid)")
    @PostMapping(Paths.FOLDER_FAVORITE)
    public ResponseEntity<Void> favoriteFolder(@PathVariable Long folderId, @RequestParam String userUuid) throws FunctionalException {
        folderService.markFolderAsFavorite(folderId, userUuid);
        return ResponseEntity.ok().build();
    }

    @PreAuthorize("@authorizationService.isTokenSubjectOwner(#requestAuthorization, #ownerUuid)")
    @DeleteMapping(Paths.FOLDER_CLEAN_FAVORITE)
    public ResponseEntity<Void> unfavoriteFolder(@PathVariable Long folderId, @RequestParam String userUuid) throws FunctionalException {
        folderService.unmarkFolderAsFavorite(folderId, userUuid);
        return ResponseEntity.noContent().build();
    }

    @PreAuthorize("@authorizationService.isTokenSubjectOwner(#requestAuthorization, #ownerUuid)")
    @GetMapping(Paths.FAVORITES)
    public ResponseEntity<List<FolderDTO>> getFavorites(@PathVariable String ownerUuid) {
        return ResponseEntity.ok(folderService.getFavoriteFolders(ownerUuid));
    }

    @PreAuthorize("@authorizationService.isTokenSubjectOwner(#requestAuthorization, #ownerUuid)")
    @PostMapping(Paths.BULK_DELETE)
    public ResponseEntity<Void> bulkDelete(@RequestBody List<Long> folderIds, @RequestParam String ownerUuid) throws FunctionalException {
        folderService.bulkDelete(folderIds, ownerUuid);
        return ResponseEntity.ok().build();
    }

    @PreAuthorize("@authorizationService.isTokenSubjectOwner(#requestAuthorization, #ownerUuid)")
    @PostMapping(Paths.BULK_SHARE)
    public ResponseEntity<Void> bulkShare(@RequestBody BulkShareRequest request) throws FunctionalException {
        folderService.bulkShare(request.getFolderIds(), request.getUserIds());
        return ResponseEntity.ok().build();
    }

    @PreAuthorize("@authorizationService.isTokenSubjectOwner(#requestAuthorization, #ownerUuid)")
    @PostMapping(Paths.BULK_MOVE)
    public ResponseEntity<Void> bulkMove(@RequestBody List<Long> folderIds, @RequestParam Long newParentId, @RequestParam String ownerUuid) throws FunctionalException {
        folderService.bulkMove(folderIds, newParentId, ownerUuid);
        return ResponseEntity.ok().build();
    }
}
