package com.marketingconfort.adanev.vsn.document.mappers;

import com.marketingconfort.adanev.vsn.document.dtos.FolderDTO;
import com.marketingconfort.adanev.vsn.document.dtos.DocumentDTO;
import com.marketingconfort.adanev.vsn.document.models.Folder;
import com.marketingconfort.adanev.vsn.document.models.Document;
import lombok.AllArgsConstructor;
import org.springframework.stereotype.Component;

import java.util.ArrayList;
import java.util.List;
import java.util.stream.Collectors;

@Component
@AllArgsConstructor
public class FolderMapper {
    private final DocumentMapper documentMapper;

    public FolderDTO toDto(Folder folder) {
        if (folder == null) return null;

        FolderDTO dto = new FolderDTO();
        dto.setId(folder.getId());
        dto.setName(folder.getName());
        dto.setOwnerUuid(folder.getOwnerUuid());
        dto.setFavorite(folder.isFavorite());
        // Set parent folder (only shallow)
        if (folder.getParent() != null) {
            FolderDTO parentDto = new FolderDTO();
            parentDto.setId(folder.getParent().getId());
            parentDto.setName(folder.getParent().getName());
            parentDto.setOwnerUuid(folder.getParent().getOwnerUuid());
            dto.setParent(parentDto);
        }
        // Conditionally map subFolders
        if (folder.getSubFolders() != null && !folder.getSubFolders().isEmpty()) {
            List<FolderDTO> subFolderDtos = folder.getSubFolders().stream()
                    .map(this::toDto) // recursive mapping (shallow by default)
                    .collect(Collectors.toList());
            dto.setSubFolders(subFolderDtos);
        }

        // Conditionally map documents
        if (folder.getDocuments() != null && !folder.getDocuments().isEmpty()) {
            List<DocumentDTO> documentDtos = folder.getDocuments().stream()
                    .map(documentMapper::toDto) // prevent infinite nesting
                    .collect(Collectors.toList());
            dto.setDocuments(documentDtos);
        }

        if (folder.getSharedWith() != null && !folder.getSharedWith().isEmpty()) {
            dto.setSharedWith(folder.getSharedWith());
        }

        return dto;
    }

    public static Folder toEntity(FolderDTO dto, Folder parent) {
        if (dto == null) return null;

        Folder folder = new Folder();
        folder.setId(dto.getId());
        folder.setName(dto.getName());
        folder.setOwnerUuid(dto.getOwnerUuid());
        folder.setParent(parent);
        folder.setSharedWith(dto.getSharedWith() != null ? dto.getSharedWith() : new ArrayList<>());
        folder.setFavorite(dto.isFavorite());

        // ✔ Map subFolders recursively
        if (dto.getSubFolders() != null && !dto.getSubFolders().isEmpty()) {
            List<Folder> subFolders = dto.getSubFolders().stream()
                    .map(subDto -> toEntity(subDto, folder)) // recursively set parent
                    .collect(Collectors.toList());
            folder.setSubFolders(subFolders);
        } else {
            folder.setSubFolders(new ArrayList<>());
        }

        // ✔ Map documents
        if (dto.getDocuments() != null && !dto.getDocuments().isEmpty()) {
            List<Document> documents = dto.getDocuments().stream()
                    .map(docDto -> DocumentMapper.toEntity(docDto, folder)) // map with parent
                    .collect(Collectors.toList());
            folder.setDocuments(documents);
        } else {
            folder.setDocuments(new ArrayList<>());
        }

        return folder;
    }

}
