package com.marketingconfort.adanev.vsn.document.services.Implementations;

import com.amazonaws.services.s3.AmazonS3;
import com.amazonaws.services.s3.model.BucketLifecycleConfiguration;
import com.amazonaws.services.s3.model.ObjectMetadata;
import com.marketingconfort.adanev.vsn.document.config.AWSConfig;
import com.marketingconfort.adanev.vsn.document.exceptions.enums.FunctionalExceptionType;
import com.marketingconfort.starter.core.exceptions.FunctionalException;
import org.springframework.stereotype.Service;
import org.springframework.web.multipart.MultipartFile;

import java.io.IOException;
import java.util.Calendar;
import java.util.Collections;
@Service
public class DocumentStorageService {
    private static final AmazonS3 amazonS3Client = AWSConfig.getInstance().amazonS3();
    private static DocumentStorageService instance = null;
    private static final int FILE_SHAREABLE_LINK_EXPIRATION_TIME = 7 * 24 * 60 * 60 * 1000;

    public static DocumentStorageService getInstance() {
        if (instance == null)
            instance = new DocumentStorageService();
        return instance;
    }

    public void createBucket(String bucketName) {
        if (!isBucketFound(bucketName)) {
            amazonS3Client.createBucket(bucketName);
            BucketLifecycleConfiguration.Rule objectsExpirationRule = new BucketLifecycleConfiguration.Rule()
                    .withExpirationInDays(1)
                    .withStatus(BucketLifecycleConfiguration.ENABLED);
            BucketLifecycleConfiguration bucketLifecycleConfiguration = new BucketLifecycleConfiguration()
                    .withRules(Collections.singletonList(objectsExpirationRule));
            amazonS3Client.setBucketLifecycleConfiguration(bucketName, bucketLifecycleConfiguration);
        }
    }

    public void uploadFile(String bucketName, String s3Key, MultipartFile file) throws FunctionalException, IOException {
        createBucket(bucketName);

        ObjectMetadata objectMetadata = new ObjectMetadata();
        objectMetadata.setContentType(file.getContentType());
        objectMetadata.setContentLength(file.getSize());

        amazonS3Client.putObject(
                bucketName,
                s3Key,
                file.getInputStream(),
                objectMetadata
        );
    }

    public String getFileSharableLink(String bucketName, String filename) throws FunctionalException {
        Calendar calendar = Calendar.getInstance();
        calendar.setTimeInMillis(calendar.getTimeInMillis() + FILE_SHAREABLE_LINK_EXPIRATION_TIME);
        if (!isBucketFound(bucketName))
            throw new FunctionalException(String.valueOf(FunctionalExceptionType.BUCKET_NOT_FOUND));
        return amazonS3Client.generatePresignedUrl(bucketName, filename, calendar.getTime()).toString();
    }

    public boolean isBucketFound(String bucketName) {
        return amazonS3Client.doesBucketExistV2(bucketName);
    }
}

