package com.marketingconfort.adanev.vsn.document.services.Implementations;

import com.marketingconfort.adanev.vsn.document.constants.MessageConstants;
import com.marketingconfort.adanev.vsn.document.dtos.DocumentDTO;
import com.marketingconfort.adanev.vsn.document.dtos.FolderDTO;
import com.marketingconfort.adanev.vsn.document.dtos.requests.NewFolderRequest;
import com.marketingconfort.adanev.vsn.document.dtos.requests.ShareFolderRequest;
import com.marketingconfort.adanev.vsn.document.dtos.response.FolderDetailsResponse;
import com.marketingconfort.adanev.vsn.document.dtos.response.FolderSizeResponseDTO;
import com.marketingconfort.adanev.vsn.document.mappers.DocumentMapper;
import com.marketingconfort.adanev.vsn.document.mappers.FolderMapper;
import com.marketingconfort.adanev.vsn.document.models.Document;
import com.marketingconfort.adanev.vsn.document.models.Folder;
import com.marketingconfort.adanev.vsn.document.repositories.DocumentRepository;
import com.marketingconfort.adanev.vsn.document.repositories.FolderRepository;
import com.marketingconfort.adanev.vsn.document.services.FolderService;
import com.marketingconfort.adanev.vsn.document.utils.FolderSizeCalculator;
import com.marketingconfort.adanev.vsn.document.utils.FolderUtils;
import com.marketingconfort.starter.core.exceptions.FunctionalException;
import jakarta.persistence.criteria.Join;
import jakarta.persistence.criteria.JoinType;
import jakarta.persistence.criteria.Predicate;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.data.domain.*;
import org.springframework.data.jpa.domain.Specification;
import org.springframework.http.ContentDisposition;
import org.springframework.http.HttpHeaders;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.StreamUtils;


import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.stream.Collectors;
import java.util.zip.ZipEntry;
import java.util.zip.ZipOutputStream;

@Service
@Slf4j
@RequiredArgsConstructor
public class FolderServiceImpl implements FolderService {
    private final FolderRepository folderRepository;
    private final DocumentRepository documentRepository;
    private final DocumentMapper documentMapper;
    private final FolderMapper folderMapper;

    @Override
    @Transactional(rollbackFor = FunctionalException.class)
    public FolderDTO createFolder(NewFolderRequest request) throws FunctionalException {
        Folder parent = validateFolderData(request);
        Folder newFolder = new Folder();
        newFolder.setName(request.getName());
        newFolder.setParent(parent);
        newFolder.setOwnerUuid(request.getOwnerUuid());
        Folder saved = folderRepository.save(newFolder);
        if (parent != null) {
            parent.getSubFolders().add(saved);
            folderRepository.save(parent);
        }
        return folderMapper.toDto(saved);
    }


    @Override
    @Transactional(rollbackFor = {FunctionalException.class})
    public FolderDTO renameFolder(Long id, String newName, String ownerUuid) throws FunctionalException {
        Folder folder = folderRepository.findById(id)
                .orElseThrow(() -> new FunctionalException(MessageConstants.FOLDER_NOT_FOUND));
        if (!folder.getOwnerUuid().equals(ownerUuid)) {
            throw new FunctionalException(MessageConstants.FOLDER_ACTION_DENIED.replace("${do}", "rename"));
        }
        // Check if sibling already has same name
        validateUniqueFolderName(newName, folder.getOwnerUuid(),folder.getParent());
        folder.setName(newName);
        Folder updated = folderRepository.save(folder);

        return folderMapper.toDto(updated);
    }

    @Override
    @Transactional(rollbackFor = FunctionalException.class)
    public void deleteFolderById(Long folderId, String ownerUuid) throws FunctionalException {
        Folder folder = folderRepository.findById(folderId)
                .orElseThrow(() -> new FunctionalException(MessageConstants.FOLDER_NOT_FOUND));

        if (!folder.getOwnerUuid().equals(ownerUuid)) {
            throw new FunctionalException(MessageConstants.FOLDER_ACTION_DENIED.replace("${do}", "rename"));
        }

        // Remove reference from parent
        Folder parent = folder.getParent();
        if (parent != null) {
            parent.getSubFolders().removeIf(f -> f.getId() == folderId);
            folderRepository.save(parent); // optional but good for memory model sync
        }

        deleteFolderRecursively(folder);
    }

    @Override
    @Transactional(readOnly = true)
    public List<FolderDTO> listRootFolders(String ownerUuid) {
        List<Folder> roots = folderRepository.findByOwnerUuidAndParentIsNull(ownerUuid);
        return roots.stream().map(folderMapper::toDto).toList();
    }
    @Override
    @Transactional(readOnly = true)
    public List<FolderDTO> listSubFolders(Long parentId, String ownerUuid) {
        List<Folder> subFolders = folderRepository.findByParentIdAndOwnerUuid(parentId, ownerUuid);
        return subFolders.stream().map(folderMapper::toDto).toList();
    }

    @Override
    @Transactional(readOnly = true)
    public List<DocumentDTO> listDocumentsInFolder(Long folderId, String ownerUuid) {
        List<Document> docs = documentRepository.findByFolderIdAndOwnerUuid(folderId, ownerUuid);
        return docs.stream().map(documentMapper::toDto).toList();
    }

    @Override
    @Transactional(rollbackFor = FunctionalException.class)
    public FolderDTO moveFolder(Long folderId, Long newParentId, String ownerUuid) throws FunctionalException {
        Folder folder = folderRepository.findById(folderId)
                .orElseThrow(() -> new FunctionalException(MessageConstants.FOLDER_NOT_FOUND));

        if (!folder.getOwnerUuid().equals(ownerUuid)) {
            throw new FunctionalException(MessageConstants.FOLDER_ACTION_DENIED.replace("${do}", "move"));
        }

        Folder oldParent = folder.getParent();

        Folder newParent = null;
        if (newParentId != null) {
            newParent = folderRepository.findById(newParentId)
                    .orElseThrow(() -> new FunctionalException(MessageConstants.PARENT_FOLDER_NOT_FOUND));
        }

        // can't move into itself or its own descendant
        if (newParent != null && (folderId.equals(newParentId) || isDescendant(folder, newParent))) {
            throw new FunctionalException(MessageConstants.MOVE_FOLDER_TO_SUBFOLDER);
        }

        //Remove from old parent's subFolders
        if (oldParent != null) {
            oldParent.getSubFolders().removeIf(f -> f.getId() == folderId);
            folderRepository.save(oldParent);
        }

        //Set new parent (null = move to root)
        folder.setParent(newParent);
        folderRepository.save(folder);

        // Add to new parent's subFolders
        if (newParent != null) {
            newParent.getSubFolders().add(folder);
            folderRepository.save(newParent);
        }

        return folderMapper.toDto(folder);
    }

    @Override
    public List<FolderDTO> searchFolders(String keyword, String ownerUuid) {
        Specification<Folder> spec = (root, query, cb) -> {
            query.distinct(true); // Avoid duplicate results due to joins

            Join<Object, Object> docJoin = root.join("documents", JoinType.LEFT);

            Predicate folderNameMatch = cb.like(cb.lower(root.get("name")), "%" + keyword.toLowerCase() + "%");
            Predicate docNameMatch = cb.like(cb.lower(docJoin.get("name")), "%" + keyword.toLowerCase() + "%");
            Predicate ownerMatch = cb.equal(root.get("ownerUuid"), ownerUuid);

            return cb.and(ownerMatch, cb.or(folderNameMatch, docNameMatch));
        };

        List<Folder> folders = folderRepository.findAll(spec);
        return folders.stream().map(folderMapper::toDto).collect(Collectors.toList());
    }


    @Override
    public FolderDetailsResponse getFolderDetails(Long folderId) throws FunctionalException {
        Folder folder = folderRepository.findById(folderId)
                .orElseThrow(() -> new FunctionalException(MessageConstants.FOLDER_NOT_FOUND));

        int docCount = folder.getDocuments() != null ? folder.getDocuments().size() : 0;
        int subCount = folder.getSubFolders() != null ? folder.getSubFolders().size() : 0;

        long totalSize = folder.getDocuments() != null
                ? folder.getDocuments().stream().mapToLong(Document::getSize).sum()
                : 0;

        return new FolderDetailsResponse(
                folder.getId(),
                folder.getName(),
                totalSize,
                docCount,
                subCount,
                folder.getCreatedAt().toLocalDate()
        );
    }

    @Override
    public Page<FolderDTO> advancedSearchFolders(String keyword, String ownerUuid, int page, int size, String sortBy, String sortDirection) throws FunctionalException {
        try {
            Sort sort = Sort.by(Sort.Direction.fromString(sortDirection), sortBy);
            Pageable pageable = PageRequest.of(page, size, sort);

            log.info("Searching folders with parameters: keyword={}, ownerUuid={}, page={}, size={}, sortBy={}, direction={}",
                    keyword, ownerUuid, page, size, sortBy, sortDirection);

            Page<Folder> folderPage = folderRepository.searchFoldersByKeywordAndOwnerUuid(keyword.toLowerCase(), ownerUuid, pageable);

            List<FolderDTO> folderDtos = folderPage.getContent()
                    .stream()
                    .map(folderMapper::toDto)
                    .collect(Collectors.toList());

            return new PageImpl<>(folderDtos, pageable, folderPage.getTotalElements());

        } catch (Exception e) {
            log.error(MessageConstants.ERROR_SEARCHING_FOLDERS, e.getMessage(), e);
            throw new FunctionalException(MessageConstants.ERROR_SEARCHING_FOLDERS + e.getMessage());
        }
    }


    @Override
    @Transactional
    public void shareFolderWithUsers(ShareFolderRequest request) throws FunctionalException {
        Folder originalFolder = folderRepository.findById(request.getFolderId())
                .orElseThrow(() -> new FunctionalException(MessageConstants.FOLDER_NOT_FOUND));

        for (Long userId : request.getUserIds()) {
            if (!originalFolder.getSharedWith().contains(userId)) {
                originalFolder.getSharedWith().add(userId);

                // Get or create "Shared With Me" folder for this user
                Folder sharedWithMeFolder = folderRepository
                        .findByNameAndOwnerId("Shared With Me", userId)
                        .orElseGet(() -> {
                            Folder f = new Folder();
                            f.setName("Shared With Me");
                            f.setOwnerId(userId);
                            f.setSubFolders(new ArrayList<>());
                            return folderRepository.save(f);
                        });

                // Create a shallow copy of the folder
                Folder sharedCopy = new Folder();
                sharedCopy.setName(originalFolder.getName());
                sharedCopy.setOwnerId(userId); // the user receiving the folder
                sharedCopy.setParent(sharedWithMeFolder);

                // Optional: shallow copy of documents (without duplicating files)
                if (originalFolder.getDocuments() != null) {
                    List<Document> copiedDocs = originalFolder.getDocuments().stream().map(doc -> {
                        Document copy = new Document();
                        copy.setName(doc.getName());
                        copy.setSize(doc.getSize());
                        copy.setDocumentType(doc.getDocumentType());
                        copy.setPath(doc.getPath()); // reference same file
                        copy.setFolder(sharedCopy);  // set to new folder
                        return copy;
                    }).collect(Collectors.toList());
                    sharedCopy.setDocuments(copiedDocs);
                }

                folderRepository.save(sharedCopy); // this also saves the docs if Cascade is set
            }
        }

        folderRepository.save(originalFolder); // update sharedWith list
    }

    @Override
    @Transactional
    public void unshareFolderWithUser(Long folderId, String userUuid) throws FunctionalException {
        Folder originalFolder = folderRepository.findById(folderId)
                .orElseThrow(() -> new FunctionalException(MessageConstants.FOLDER_NOT_FOUND));

        // Step 1: Remove userUuid from original folder's sharedWith list
        originalFolder.getSharedWith().removeIf(uuid -> uuid.equals(userUuid));
        folderRepository.save(originalFolder);

        // Step 2: Get the "Shared With Me" folder of the user
        Folder sharedWithMeFolder = folderRepository
                .findByNameAndOwnerUuid("Shared With Me", userUuid)
                .orElseThrow(() -> new FunctionalException(MessageConstants.SHARE_WITH_ME_FOLDER_NOT_FOUND +  userUuid));

        // Step 3: Find the shared copy by matching name (optional: match path/hash/id if tracked)
        List<Folder> subFolders = sharedWithMeFolder.getSubFolders();
        if (subFolders != null && !subFolders.isEmpty()) {
            Folder toRemove = subFolders.stream()
                    .filter(sub -> sub.getName().equals(originalFolder.getName()))
                    .findFirst()
                    .orElse(null);

            if (toRemove != null) {
                subFolders.remove(toRemove); // remove from list
                folderRepository.delete(toRemove); // delete copy
            }
        }
    }

    @Override
    public byte[] zipFolderAndReturn(Long folderId) throws FunctionalException {
        Folder folder = folderRepository.findById(folderId)
                .orElseThrow(() -> new FunctionalException(MessageConstants.FOLDER_NOT_FOUND));

        try {
            ByteArrayOutputStream baos = new ByteArrayOutputStream();
            ZipOutputStream zipOut = new ZipOutputStream(baos);
            if (folder.getDocuments() != null) {
                for (Document document : folder.getDocuments()) {
                    FolderUtils.addToZip(folder.getName() + "/" + document.getName(), document.getPath(), zipOut);
                }
            }
            if (folder.getSharedWith() != null) {
                for (Folder subFolder : folder.getSubFolders()) {
                    FolderUtils.addFolderToZip(subFolder, folder.getName(), zipOut);
                }
            }

            zipOut.close();
            baos.close();

            return baos.toByteArray();

        } catch (IOException e) {
            throw new FunctionalException(MessageConstants.FOLDER_ZIP_ERROR + e.getMessage());
        }
    }

    @Override
    public ResponseEntity<byte[]> downloadZippedFolder(Long folderId) throws FunctionalException {
        Folder folder = folderRepository.findById(folderId)
                .orElseThrow(() -> new FunctionalException(MessageConstants.FOLDER_NOT_FOUND));

        byte[] zipBytes = zipFolderAndReturn(folderId);

        HttpHeaders headers = new HttpHeaders();
        headers.setContentDisposition(ContentDisposition.attachment().filename(folder.getName() + ".zip").build());
        headers.setContentType(MediaType.APPLICATION_OCTET_STREAM);
        headers.setContentLength(zipBytes.length);

        return ResponseEntity.ok()
                .headers(headers)
                .body(zipBytes);
    }

    @Override
    public FolderSizeResponseDTO calculateFolderSize(Long folderId) throws FunctionalException {
        Folder folder = folderRepository.findById(folderId)
                .orElseThrow(() -> new FunctionalException(MessageConstants.FOLDER_NOT_FOUND));

        long totalSizeBytes = FolderSizeCalculator.calculateFolderSize(folderMapper.toDto(folder));
        String formattedSize = FolderSizeCalculator.formatBytes(totalSizeBytes);

        return FolderSizeResponseDTO.builder()
                .folderId(folderId)
                .folderName(folder.getName())
                .totalSizeBytes(totalSizeBytes)
                .formattedSize(formattedSize)
                .build();
    }

    @Override
    @Transactional
    public void markFolderAsFavorite(Long folderId, String userUuid) throws FunctionalException {
        Folder folder = folderRepository.findById(folderId)
                .orElseThrow(() -> new FunctionalException(MessageConstants.FOLDER_NOT_FOUND));

        if (!folder.getOwnerUuid().equals(userUuid)) {
            throw new FunctionalException(MessageConstants.FOLDER_ACTION_DENIED.replace("${do}", "mark as favorite"));
        }

        folder.setFavorite(true);
        folderRepository.save(folder);
    }

    @Override
    @Transactional
    public void unmarkFolderAsFavorite(Long folderId, String userUuid) throws FunctionalException {
        Folder folder = folderRepository.findById(folderId)
                .orElseThrow(() -> new FunctionalException(MessageConstants.FOLDER_NOT_FOUND));

        if (!folder.getOwnerUuid().equals(userUuid)) {
            throw new FunctionalException(MessageConstants.FOLDER_ACTION_DENIED.replace("${do}", "unmark favorite"));
        }

        folder.setFavorite(false);
        folderRepository.save(folder);
    }

    @Override
    public List<FolderDTO> getFavoriteFolders(String userUuid) {
        List<Folder> favorites = folderRepository.findByOwnerUuidAndFavoriteTrue(userUuid);
        return favorites.stream().map(folderMapper::toDto).collect(Collectors.toList());
    }

    @Override
    @Transactional
    public void bulkDelete(List<Long> folderIds, String ownerUuid) throws FunctionalException {
        for (Long folderId : folderIds) {
            deleteFolderById(folderId, ownerUuid);
        }
    }
    @Override
    @Transactional
    public void bulkShare(List<Long> folderIds, List<Long> userIds) throws FunctionalException {
        for (Long folderId : folderIds) {
            shareFolderWithUsers(new ShareFolderRequest(
                    folderId,
                    userIds
            ));
        }
    }
    @Override
    @Transactional
    public void bulkMove(List<Long> folderIds, Long newParentId, String ownerUuid) throws FunctionalException {
        for (Long folderId : folderIds) {
            moveFolder(folderId, newParentId, ownerUuid);
        }
    }

    //helpers
    private boolean isDescendant(Folder folder, Folder targetParent) {
        Folder current = targetParent;
        while (current != null) {
            if (current.getId() == folder.getId()) {
                return true;
            }
            current = current.getParent();
        }
        return false;
    }

    private void deleteFolderRecursively(Folder folder) {
        // Step 1: delete subfolders first (depth-first)
        if (folder.getSubFolders() != null && !folder.getSubFolders().isEmpty()) {
            for (Folder sub : folder.getSubFolders()) {
                deleteFolderRecursively(sub);
            }
        }

        // Step 2: delete documents inside this folder
        if (folder.getDocuments() != null && !folder.getDocuments().isEmpty()) {
            for (Document doc : folder.getDocuments()) {
                documentRepository.delete(doc);
            }
            folder.getDocuments().clear();
        }

        // Step 3: finally delete the folder
        folder.getSubFolders().clear(); // ensure memory consistency
        folderRepository.delete(folder);
    }

    private Folder validateFolderData(NewFolderRequest request) throws FunctionalException {
        Folder parent = null;
        if (request.getParentId() != null) {
            parent =  folderRepository.findById(request.getParentId())
                    .orElseThrow(() -> new FunctionalException(MessageConstants.PARENT_FOLDER_NOT_FOUND));
        }
        if (parent != null) {
            validateUniqueFolderName(request.getName(), request.getOwnerUuid(), parent);
        }
        return parent;
    }
    //checks the folder name
    private void validateUniqueFolderName(String name, String ownerUuid, Folder parent) throws FunctionalException {
        boolean exists = folderRepository.existsByNameAndOwnerUuidAndParent(name, ownerUuid, parent);
        boolean isParentName = false;
        if (parent != null) {
            isParentName = name.equals(parent.getName());
        }
        if (exists) {
            throw new FunctionalException(MessageConstants.FOLDER_ALREADY_EXISTS);
        }
        if (isParentName) {
            throw new FunctionalException(MessageConstants.FOLDER_PARENT_NAME);
        }
    }

}
