package com.marketingconfort.adanev.vsn.document.services.Implementations;

import com.marketingconfort.adanev.vsn.document.constants.MessageConstants;
import com.marketingconfort.adanev.vsn.document.constants.StorageQuotaConstants;
import com.marketingconfort.adanev.vsn.document.dtos.StorageQuotaInfoDTO;
import com.marketingconfort.adanev.vsn.document.models.Document;
import com.marketingconfort.adanev.vsn.document.models.Folder;
import com.marketingconfort.adanev.vsn.document.repositories.DocumentRepository;
import com.marketingconfort.adanev.vsn.document.repositories.FolderRepository;
import com.marketingconfort.adanev.vsn.document.services.StorageQuotaService;
import com.marketingconfort.adanev.vsn.document.utils.FolderSizeCalculator;
import com.marketingconfort.adanev.vsn.document.mappers.FolderMapper;
import com.marketingconfort.adanev.vsn.document.utils.StorageQuotaUtils;
import com.marketingconfort.starter.core.exceptions.FunctionalException;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.time.LocalDateTime;
import java.util.List;

@Service
@Slf4j
@RequiredArgsConstructor
public class StorageQuotaServiceImpl implements StorageQuotaService {

    private final FolderRepository folderRepository;
    private final DocumentRepository documentRepository;
    private final FolderMapper folderMapper;
    @Override
    @Transactional(readOnly = true)
    public StorageQuotaInfoDTO getStorageQuotaInfo(Long ownerId) throws FunctionalException {
        if (ownerId == null) {
            throw new FunctionalException(MessageConstants.NULL_OWNER_ID);
        }

        // Calculate total used storage from all folders owned by this owner
        Long usedStorage = calculateTotalUsedStorage(ownerId);

        Long availableStorage = Math.max(0L, StorageQuotaConstants.STORAGE_QUOTA_BYTES - usedStorage);
        double usagePercentage = StorageQuotaConstants.STORAGE_QUOTA_BYTES > 0 ? (double) usedStorage / StorageQuotaConstants.STORAGE_QUOTA_BYTES * 100 : 0.0;
        boolean isQuotaExceeded = usedStorage > StorageQuotaConstants.STORAGE_QUOTA_BYTES;

        // Count total folders owned by this user
        List<Folder> ownedFolders = folderRepository.findByOwnerId(ownerId);
        int totalFolders = ownedFolders.size();

        return StorageQuotaInfoDTO.builder()
                .ownerId(ownerId)
                .storageQuotaBytes(StorageQuotaConstants.STORAGE_QUOTA_BYTES)
                .storageQuotaFormatted(StorageQuotaConstants.STORAGE_QUOTA_FORMATTED)
                .usedStorageBytes(usedStorage)
                .usedStorageFormatted(FolderSizeCalculator.formatBytes(usedStorage))
                .availableStorageBytes(availableStorage)
                .availableStorageFormatted(FolderSizeCalculator.formatBytes(availableStorage))
                .usagePercentage(Math.round(usagePercentage * 100.0) / 100.0)
                .isQuotaExceeded(isQuotaExceeded)
                .totalFolders(totalFolders)
                .lastCalculated(LocalDateTime.now())
                .build();
    }

    @Override
    @Transactional(readOnly = true)
    public boolean hasAvailableStorage(Long ownerId, Long additionalSizeBytes) throws FunctionalException {
        if (ownerId == null || additionalSizeBytes == null) {
            throw new FunctionalException(MessageConstants.NULL_VALUE);
        }

        Long currentUsedStorage = calculateTotalUsedStorage(ownerId);
        return (currentUsedStorage + additionalSizeBytes) <= StorageQuotaConstants.STORAGE_QUOTA_BYTES;
    }

    @Override
    @Transactional(readOnly = true)
    public boolean isStorageQuotaExceeded(Long ownerId) throws FunctionalException {
        if (ownerId == null) {
            throw new FunctionalException(MessageConstants.NULL_OWNER_ID);
        }

        Long usedStorage = calculateTotalUsedStorage(ownerId);
        return StorageQuotaUtils.isExceeded(usedStorage);
    }

    @Override
    @Transactional(readOnly = true)
    public double getStorageUsagePercentage(Long ownerId) throws FunctionalException {
        if (ownerId == null) {
            throw new FunctionalException(MessageConstants.NULL_OWNER_ID);
        }

        Long usedStorage = calculateTotalUsedStorage(ownerId);

        if (StorageQuotaConstants.STORAGE_QUOTA_BYTES == 0) {
            return 0.0;
        }

        return Math.round((double) usedStorage / StorageQuotaConstants.STORAGE_QUOTA_BYTES * 100.0 * 100.0) / 100.0;
    }

    @Override
    @Transactional(readOnly = true)
    public Long getTotalUsedStorage(Long ownerId) throws FunctionalException {
        if (ownerId == null) {
            throw new FunctionalException(MessageConstants.NULL_OWNER_ID);
        }

        return calculateTotalUsedStorage(ownerId);
    }

    @Override
    @Transactional(readOnly = true)
    public Long getAvailableStorage(Long ownerId) throws FunctionalException {
        if (ownerId == null) {
            throw new FunctionalException(MessageConstants.NULL_OWNER_ID);
        }

        Long usedStorage = calculateTotalUsedStorage(ownerId);
        return Math.max(0L, StorageQuotaConstants.STORAGE_QUOTA_BYTES - usedStorage);
    }

    private Long calculateTotalUsedStorage(Long ownerId) {
        List<Folder> ownedFolders = folderRepository.findByOwnerId(ownerId);
        List<Document> rootDocuments = documentRepository.findByOwnerIdAndFolderIsNull(ownerId);

        long foldersSize = ownedFolders.stream()
                .mapToLong(folder -> FolderSizeCalculator.calculateFolderSize(folderMapper.toDto(folder)))
                .sum();

        long rootDocsSize = rootDocuments.stream()
                .filter(doc -> doc.getSize() != null)
                .mapToLong(Document::getSize)
                .sum();

        return foldersSize + rootDocsSize;
    }



    public static Long getStorageQuotaBytes() {
        return StorageQuotaConstants.STORAGE_QUOTA_BYTES;
    }

    public static String getStorageQuotaFormatted() {
        return StorageQuotaConstants.STORAGE_QUOTA_FORMATTED;
    }
}
