import type { IAIAssistantTableFilters } from 'src/types/ai-assistant';
import { useMemo } from 'react';
import Box from '@mui/material/Box';
import Chip from '@mui/material/Chip';
import Stack from '@mui/material/Stack';
import Button from '@mui/material/Button';
import Typography from '@mui/material/Typography';
import { Iconify } from 'src/shared/components/iconify';

//----------------------------------------------------------------------
type Props = {
  filters: IAIAssistantTableFilters;
  onFilters: (name: string, value: string | string[]) => void;
  onResetFilters: () => void;
  onResetPage: () => void; // Add this line

  sx?: object; // Ajoutez cette ligne pour accepter la prop sx
  results: number;
  options: {
    types: {
      value: string;
      label: string;
    }[];
    educationLevels: {
      value: string;
      label: string;
    }[];
    subjects: {
      value: string;
      label: string;
    }[];
    statuses: {
      value: string;
      label: string;
    }[];
  };
};

export function AIAssistantTableFiltersResult({
  filters,
  onFilters,
  onResetFilters,
  results,
  options,
}: Props) {
  // Vérification si le type "J'apprends" est sélectionné
  const isJapprendsSelected = useMemo(() => {
    // Supposons que la valeur du type "J'apprends" soit "japprends"
    // Vous devrez ajuster cette valeur selon votre configuration réelle
    return filters.type.includes('japprends');
  }, [filters.type]);

  const handleRemoveType = (value: string) => {
    const newValue = filters.type.filter((item) => item !== value);
    onFilters('type', newValue);
  };

  const handleRemoveFilter = (name: string) => {
    onFilters(name, '');
  };

  // Calcul des filtres actifs et à afficher
  const activeFilters = useMemo(() => {
    // Filtres toujours visibles
    const baseFilters: { name: string, label: string, value: string | string[], valueLabel?: string }[] = [];
    
    // Ajouter les types sélectionnés
    if (filters.type.length) {
      filters.type.forEach((typeValue) => {
        const typeOption = options.types.find((option) => option.value === typeValue);
        if (typeOption) {
          baseFilters.push({
            name: 'type',
            label: 'Type',
            value: typeValue,
            valueLabel: typeOption.label,
          });
        }
      });
    }
    
    // Ajouter le niveau d'éducation s'il est défini
    if (filters.educationLevel) {
      const levelOption = options.educationLevels.find((option) => option.value === filters.educationLevel);
      baseFilters.push({
        name: 'educationLevel',
        label: "Niveau d'éducation",
        value: filters.educationLevel,
        valueLabel: levelOption?.label,
      });
    }
    
    // Ajouter le statut s'il est défini
    if (filters.status) {
      const statusOption = options.statuses.find((option) => option.value === filters.status);
      baseFilters.push({
        name: 'status',
        label: 'Statut',
        value: filters.status,
        valueLabel: statusOption?.label,
      });
    }
    
    // Ajouter le nom s'il est défini
    if (filters.name) {
      baseFilters.push({
        name: 'name',
        label: 'Nom',
        value: filters.name,
      });
    }
    
    // Ajouter les filtres spécifiques à "J'apprends" seulement si ce type est sélectionné
    if (isJapprendsSelected) {
      // Ajouter la matière si elle est définie
      if (filters.subject) {
        const subjectOption = options.subjects.find((option) => option.value === filters.subject);
        baseFilters.push({
          name: 'subject',
          label: 'Matière',
          value: filters.subject,
          valueLabel: subjectOption?.label,
        });
      }
      
      // Ajouter le chapitre s'il est défini
      if (filters.chapter) {
        baseFilters.push({
          name: 'chapter',
          label: 'Chapitre',
          value: filters.chapter,
        });
      }
      
      // Ajouter l'exercice s'il est défini
      if (filters.exercise) {
        baseFilters.push({
          name: 'exercise',
          label: 'Exercice',
          value: filters.exercise,
        });
      }
    }
    
    return baseFilters;
  }, [filters, options, isJapprendsSelected]);

  // Ne rien afficher s'il n'y a pas de filtres actifs
  if (!activeFilters.length) {
    return null;
  }

  return (
    <Stack spacing={1.5} sx={{ mb: 3 }}>
      <Stack direction="row" alignItems="center" sx={{ typography: 'body2' }}>
        <Box component="span" sx={{ typography: 'subtitle2' }}>
          {results}
        </Box>

        <Box component="span" sx={{ ml: 0.25 }}>
          résultats trouvés
        </Box>

        {activeFilters.length > 0 && (
          <>
            <Box component="span" sx={{ ml: 0.25 }}>
              avec
            </Box>

            <Box component="span" sx={{ ml: 0.5, typography: 'subtitle2' }}>
              {activeFilters.length} filtres
            </Box>
          </>
        )}
      </Stack>

      <Stack direction="row" flexWrap="wrap" alignItems="center" spacing={1}>
        {activeFilters.map((filter) => {
          const getChipLabel = () => {
            if (filter.name === 'type') {
              return filter.valueLabel;
            }
            if (filter.valueLabel) {
              return `${filter.label}: ${filter.valueLabel}`;
            }
            return `${filter.label}: ${filter.value}`;
          };

          return (
            <Chip
              key={`${filter.name}-${filter.value}`}
              label={getChipLabel()}
              size="small"
              variant="soft"
              color="primary"
              onDelete={
                filter.name === 'type'
                  ? () => handleRemoveType(filter.value as string)
                  : () => handleRemoveFilter(filter.name)
              }
            />
          );
        })}

        <Button
          color="error"
          onClick={onResetFilters}
          startIcon={<Iconify icon="solar:trash-bin-trash-bold" />}
        >
          Effacer
        </Button>
      </Stack>
    </Stack>
  );
}