import type { IAIAssistantItem } from 'src/types/ai-assistant';
import TableRow from '@mui/material/TableRow';
import TableCell from '@mui/material/TableCell';
import IconButton from '@mui/material/IconButton';
import Tooltip from '@mui/material/Tooltip';
import Chip from '@mui/material/Chip';
import Box from '@mui/material/Box';
import Typography from '@mui/material/Typography';
import { Iconify } from 'src/shared/components/iconify';

type Props = {
  row: IAIAssistantItem;
  selected: boolean;
  onSelectRow: VoidFunction;
  onEditRow: VoidFunction;
  onSettingsRow : VoidFunction;
  onDeleteRow: VoidFunction;
  onOpenDeleteConfirm: VoidFunction;
};

export function AIAssistantTableRow({
  row,
  selected,
  onSelectRow,
  onDeleteRow,
  onEditRow,
  onSettingsRow,
  onOpenDeleteConfirm,
}: Props) {
  const { id, name, educationLevel, type, status, subject, chapter, exercise, createdAt } = row;
  const isJapprends = type === 'japprends';

  // Fonction pour afficher le statut avec une couleur appropriée
  const renderStatus = (status: string) => {
    const statusColors: Record<string, 'success' | 'warning' | 'error' | 'default'> = {
      active: 'success',
      inactive: 'error',
      draft: 'default',
    };
    
    return (
      <Chip
        size="small"
        label={status === 'active' ? 'Actif' : status === 'inactive' ? 'Inactif' : status}
        color={statusColors[status.toLowerCase()] || 'default'}
        sx={{ textTransform: 'capitalize' }}
      />
    );
  };

  // Fonction pour afficher une cellule avec un tiret si la valeur est vide/null
  const renderCell = (value: string | null) => {
    return (
      <Typography variant="body2" noWrap>
        {value || '-'}
      </Typography>
    );
  };

  return (
    <TableRow
      hover
      selected={selected}
      sx={{
        backgroundColor: 'background.paper',
        boxShadow: (theme) => theme.customShadows.z1,
        '& td': { borderBottom: 0, height: '72px', position: 'relative' },
        '& td:first-of-type': { borderTopLeftRadius: 1.5, borderBottomLeftRadius: 1.5 },
        '& td:last-of-type': { borderTopRightRadius: 1.5, borderBottomRightRadius: 1.5 },
      }}
    >
      {/* Nom */}
      <TableCell>
        <Box sx={{ display: 'flex', alignItems: 'center' }}>
          <Typography variant="body2" noWrap>
            {name}
          </Typography>
        </Box>
      </TableCell>
      
      {/* Type */}
      <TableCell>
        <Typography variant="body2" noWrap>
          {type === 'japprends' ? "J'apprends" : 
           type === 'accueil' ? "Accueil" : 
           type === 'recherche' ? "Recherche" : type}
        </Typography>
      </TableCell>
          
    {/* Date d'ajout */}
    {/* Date d'ajout */}
    <TableCell>
      <Typography variant="body2" noWrap>
        {row.createdAt && typeof row.createdAt === 'string' && row.createdAt !== "-" 
          ? new Date(row.createdAt).toLocaleDateString() 
          : '-'}
      </Typography>
    </TableCell>

      
      {/* Niveau d'Éducation */}
      <TableCell>
        {renderCell(educationLevel)}
      </TableCell>
      
      {/* Matière */}
      <TableCell>
        {renderCell(isJapprends ? subject : null)}
      </TableCell>
      
      {/* Chapitre */}
      <TableCell>
        {renderCell(isJapprends ? chapter : null)}
      </TableCell>
      
      {/* Exercice */}
      <TableCell>
        {renderCell(isJapprends ? exercise : null)}
      </TableCell>
      
      {/* Statut */}
      <TableCell>
        {renderStatus(status)}
      </TableCell>
      
      {/* Actions */}
      <TableCell align="right">
        <Box sx={{ display: 'flex', alignItems: 'center' }}>
          {/* Bouton Modifier */}
          <Tooltip title="Modifier">
            <IconButton color="primary" onClick={onEditRow}>
              <Iconify icon="solar:pen-bold" />
            </IconButton>
          </Tooltip>

          {/* Bouton Paramètres */}
          <Tooltip title="Paramètres">
            <IconButton color="secondary" onClick={onSettingsRow}>
              <Iconify icon="mdi:cog-outline" />
            </IconButton>
          </Tooltip>
        </Box>
      </TableCell>
    </TableRow>
  );
}