import { useState, useMemo, useEffect } from 'react';
import type { IAIAssistantItem, IAIAssistantTableFilters } from 'src/types/ai-assistant';
import type { TableProps } from 'src/shared/components/table';
import type { UseSetStateReturn } from 'src/hooks/use-set-state';
import { useRouter } from 'next/navigation'; // Use navigation for App Router

import Box from '@mui/material/Box';
import Table from '@mui/material/Table';
import TableBody from '@mui/material/TableBody';
import TableContainer from '@mui/material/TableContainer';
import TableRow from '@mui/material/TableRow';
import TableCell from '@mui/material/TableCell';
import TextField from '@mui/material/TextField';
import InputAdornment from '@mui/material/InputAdornment';
import { useTheme } from '@mui/material/styles';
import { tableCellClasses } from '@mui/material/TableCell';
import { tablePaginationClasses } from '@mui/material/TablePagination';
import Button from '@mui/material/Button';
import Dialog from '@mui/material/Dialog';
import DialogTitle from '@mui/material/DialogTitle';
import DialogContent from '@mui/material/DialogContent';
import DialogActions from '@mui/material/DialogActions';
import DialogContentText from '@mui/material/DialogContentText';
import IconButton from '@mui/material/IconButton';
import { useBoolean } from 'src/hooks/use-boolean';
import { useSetState } from 'src/hooks/use-set-state';

import { Iconify } from 'src/shared/components/iconify';
import {
  TableNoData,
  TableHeadCustom,
  TablePaginationCustom,
} from 'src/shared/components/table';

import { AIAssistantTableRow } from './ai-assistant-table-row';
import { AIAssistantTableToolbar } from './ai-assistant-table-toolbar';
import { AIAssistantTableFiltersResult } from './ai-assistant-table-filters-result';
import { AI_ASSISTANT_STATUS_OPTIONS, AI_ASSISTANT_TYPE_OPTIONS, AI_ASSISTANT_EDUCATION_LEVELS, AI_ASSISTANT_SUBJECTS } from 'src/shared/_mock/_ai-assistant';

import { DatePicker } from '@mui/x-date-pickers/DatePicker';
import { LocalizationProvider } from '@mui/x-date-pickers/LocalizationProvider';
import { AdapterDateFns } from '@mui/x-date-pickers/AdapterDateFns';

// Fonction utilitaire pour formater les dates en DD/MM/YYYY
const formatDate = (date: Date) => {
  const day = String(date.getDate()).padStart(2, '0');
  const month = String(date.getMonth() + 1).padStart(2, '0');
  const year = date.getFullYear();
  return `${day}/${month}/${year}`;
};

// ----------------------------------------------------------------------

// Modifié pour avoir des colonnes de tailles égales
const TABLE_HEAD = [
  { id: 'name', label: 'Nom', width: '12.5%' },
  { id: 'type', label: 'Type', width: '12.5%' },
  { id: 'createdAt', label: 'Date d\'ajout', width: 150 },
  { id: 'educationLevel', label: 'Niveau d\'Éducation', width: '12.5%' },
  { id: 'subject', label: 'Matière', width: '12.5%' },
  { id: 'chapter', label: 'Chapitre', width: '12.5%' },
  { id: 'exercise', label: 'Exercice', width: '12.5%' },
  { id: 'status', label: 'Statut', width: '12.5%' },
  { id: '', width: '12.5%' },
];

// ----------------------------------------------------------------------
type Props = {
  table: TableProps;
  notFound: boolean;
  dataFiltered: IAIAssistantItem[];
  onDeleteRow: (id: string) => void;
  onEditRow: (id: string) => void;
  onSettingsRow : (id: string) => void;
  filters?: IAIAssistantTableFilters;
  isJapprendsSelected?: boolean;
  onFilterChange?: (filterName: string, value: string | string[]) => void;
  onClearFilters?: () => void;
};

// Helper function to safely access item properties
const handleItemValue = (item: IAIAssistantItem, column: string): string => {
  switch (column) {
    case 'name': return item.name?.toString().toLowerCase() || '';
    case 'type': return item.type?.toString().toLowerCase() || '';
    case 'educationLevel': return item.educationLevel?.toString().toLowerCase() || '';
    case 'subject': return item.subject?.toString().toLowerCase() || '';
    case 'chapter': return item.chapter?.toString().toLowerCase() || '';
    case 'exercise': return item.exercise?.toString().toLowerCase() || '';
    case 'status': return item.status?.toString().toLowerCase() || '';
    default: return '';
  }
};

export function AIAssistantTable({
  table,
  notFound,
  onDeleteRow,
  onEditRow,
  onSettingsRow,
  dataFiltered,
  filters = {} as IAIAssistantTableFilters,
  isJapprendsSelected = false,
  onFilterChange = () => {},
  onClearFilters = () => {},
}: Props) {
  const theme = useTheme();
  const router = useRouter(); // ✅ Appel du Hook au bon endroit
  const handleSettings = (id: string) => {
    router.push(`/dashboard/personalization-ai/settings/${id}`);
  };
  // Créer un état local pour les filtres
  const internalFilters = useSetState({
    name: filters.name || '',
    type: filters.type || [],
    status: filters.status || '',
    educationLevel: filters.educationLevel || '',
    subject: filters.subject || '',
    chapter: filters.chapter || '',
    exercise: filters.exercise || '',
  });

  // État pour les filtres de recherche par colonne
  const [searchFilters, setSearchFilters] = useState<{ [key: string]: string }>({
    name: '',
    type: '',
    createdAt: '',
    educationLevel: '',
    subject: '',
    chapter: '',
    exercise: '',
    status: ''
  });

  // État pour les données filtrées par la recherche de colonne
  const [tableDataFiltered, setTableDataFiltered] = useState(dataFiltered);

  // Gérer les changements dans les filtres de recherche par colonne
  const handleSearchChange = (column: string, value: string) => {
    setSearchFilters((prev) => ({
      ...prev,
      [column]: value,
    }));
  };

  // Effet pour filtrer les données en fonction des filtres de recherche
  useEffect(() => {
    let filtered = [...dataFiltered];

    // Appliquer les filtres de recherche par colonne
    Object.entries(searchFilters).forEach(([column, value]) => {
      if (value) {
        if (column === 'createdAt') {
          // Comparer uniquement l'année, le mois et le jour
          filtered = filtered.filter(item =>
            item.createdAt && formatDate(new Date(item.createdAt)) === value
          );
        } else {
          const searchValue = value.toLowerCase();
          filtered = filtered.filter(item => handleItemValue(item, column).includes(searchValue));
        }
      }
    });

    setTableDataFiltered(filtered);
  }, [searchFilters, dataFiltered]);

  // Synchroniser l'état local avec les props
  useEffect(() => {
    internalFilters.setState({
      name: filters.name || '',
      type: filters.type || [],
      status: filters.status || '',
      educationLevel: filters.educationLevel || '',
      subject: filters.subject || '',
      chapter: filters.chapter || '',
      exercise: filters.exercise || '',
    });
  }, [filters]);

  // Réinitialiser la page quand les filtres changent
  useEffect(() => {
    if (page) {
      onChangePage(null, 0);
    }
  }, [searchFilters]);

  // État pour la boîte de dialogue de confirmation de suppression
  const [deleteId, setDeleteId] = useState<string | null>(null);
  const deleteDialog = useBoolean();

  const filterOptions = {
    types: AI_ASSISTANT_TYPE_OPTIONS,
    educationLevels: AI_ASSISTANT_EDUCATION_LEVELS,
    subjects: AI_ASSISTANT_SUBJECTS,
    statuses: AI_ASSISTANT_STATUS_OPTIONS,
  };

  // Fonction pour propager les changements de filtre vers le parent
  const handleFilterChange = (filterName: string, value: string | string[]) => {
    internalFilters.setState({
      [filterName]: value,
    });
    onFilterChange(filterName, value);
  };

  const handleOpenDeleteConfirm = (id: string) => {
    setDeleteId(id);
    deleteDialog.onTrue();
  };

  const handleConfirmDelete = () => {
    if (deleteId) {
      onDeleteRow(deleteId);
      deleteDialog.onFalse();
      setDeleteId(null);
    }
  };

  const {
    dense,
    page,
    order,
    orderBy,
    rowsPerPage,
    //
    onSort,
    onChangePage,
    onChangeRowsPerPage,
  } = table;

  // Fonction pour effacer un filtre de colonne spécifique
  const clearColumnFilter = (column: string) => {
    setSearchFilters(prev => ({
      ...prev,
      [column]: ''
    }));
  };

  // Fonction pour effacer tous les filtres de colonne
  const clearAllColumnFilters = () => {
    setSearchFilters({
      name: '',
      type: '',
      createdAt: '',
      educationLevel: '',
      subject: '',
      chapter: '',
      exercise: '',
      status: ''
    });
  };

  // Supposons que ces propriétés manquent dans le composant enfant
  const dummySelectRow = (id: string) => {
    console.log(`Row ${id} selected`);
    // Implémentation vide, à adapter selon vos besoins
  };
  return (
    <>
      {/* Toolbar avec filtres */}
      <AIAssistantTableToolbar
        filters={internalFilters}
        options={filterOptions}
        onResetPage={() => onChangePage(null, 0)}
        onFilterChange={handleFilterChange}
        onClearFilters={onClearFilters}
      />

      {/* Utiliser un Box wrapper pour résoudre l'erreur TypeScript */}
      <Box sx={{ mb: { xs: 3, md: 5 } }}>
        <AIAssistantTableFiltersResult
          filters={internalFilters.state}
          onFilters={onFilterChange}
          results={tableDataFiltered.length}
          onResetPage={() => onChangePage(null, 0)}
          onResetFilters={onClearFilters}
          options={filterOptions}
        />
      </Box>

      {/* Wrapper pour la table avec une hauteur fixe et scrollable */}
      <Box sx={{ position: 'relative', height: '60vh', overflow: 'hidden', border: '1px solid', borderColor: 'divider', borderRadius: 1 }}>
        {/* En-tête fixe */}
        <TableContainer sx={{ overflow: 'visible' }}>
          <Table
            size={dense ? 'small' : 'medium'}
            sx={{
              minWidth: 960,
              tableLayout: 'fixed' // Force des colonnes égales
            }}
          >
            <TableHeadCustom
              order={order}
              orderBy={orderBy}
              headLabel={TABLE_HEAD}
              onSort={onSort}
              sx={{
                [`& .${tableCellClasses.head}`]: {
                  bgcolor: 'background.paper',
                },
              }}
            />

            {/* Ligne pour les filtres de recherche par colonne */}
            <TableRow>
              <TableCell sx={{ width: '12.5%' }}>
                <TextField
                  fullWidth
                  size="small"
                  placeholder="Rechercher..."
                  value={searchFilters.name}
                  onChange={(e) => handleSearchChange('name', e.target.value)}
                  InputProps={{
                    startAdornment: (
                      <InputAdornment position="start">
                        <Iconify icon="eva:search-fill" sx={{ color: 'text.disabled' }} />
                      </InputAdornment>
                    ),
                    endAdornment: searchFilters.name && (
                      <InputAdornment position="end">
                        <IconButton size="small" onClick={() => clearColumnFilter('name')}>
                          <Iconify icon="eva:close-fill" />
                        </IconButton>
                      </InputAdornment>
                    )
                  }}
                />
              </TableCell>
              <TableCell sx={{ width: '12.5%' }}>
                <TextField
                  fullWidth
                  size="small"
                  placeholder="Type..."
                  value={searchFilters.type}
                  onChange={(e) => handleSearchChange('type', e.target.value)}
                  InputProps={{
                    startAdornment: (
                      <InputAdornment position="start">
                        <Iconify icon="eva:search-fill" sx={{ color: 'text.disabled' }} />
                      </InputAdornment>
                    ),
                    endAdornment: searchFilters.type && (
                      <InputAdornment position="end">
                        <IconButton size="small" onClick={() => clearColumnFilter('type')}>
                          <Iconify icon="eva:close-fill" />
                        </IconButton>
                      </InputAdornment>
                    )
                  }}
                />
              </TableCell>
              <TableCell sx={{ width: '12.5%' }}>
                <TextField
                  label="Date d'ajout"
                  type="date"
                  value={searchFilters.createdAt || ''}
                  onChange={(e) => {
                    handleSearchChange('createdAt', e.target.value);
                    // Réinitialiser la page si nécessaire
                    onChangePage(null, 0);
                  }}
                  InputLabelProps={{ shrink: true }}
                  fullWidth
                  size="small"
                  InputProps={{
                    endAdornment: searchFilters.createdAt && (
                      <InputAdornment position="end">
                        <IconButton 
                          size="small" 
                          onClick={() => {
                            clearColumnFilter('createdAt');
                            onChangePage(null, 0);
                          }}
                        >
                          <Iconify icon="eva:close-fill" />
                        </IconButton>
                      </InputAdornment>
                    )
                  }}
                />
              </TableCell>
              <TableCell sx={{ width: '12.5%' }}>
                <TextField
                  fullWidth
                  size="small"
                  placeholder="Niveau..."
                  value={searchFilters.educationLevel}
                  onChange={(e) => handleSearchChange('educationLevel', e.target.value)}
                  InputProps={{
                    startAdornment: (
                      <InputAdornment position="start">
                        <Iconify icon="eva:search-fill" sx={{ color: 'text.disabled' }} />
                      </InputAdornment>
                    ),
                    endAdornment: searchFilters.educationLevel && (
                      <InputAdornment position="end">
                        <IconButton size="small" onClick={() => clearColumnFilter('educationLevel')}>
                          <Iconify icon="eva:close-fill" />
                        </IconButton>
                      </InputAdornment>
                    )
                  }}
                />
              </TableCell>
              <TableCell sx={{ width: '12.5%' }}>
                <TextField
                  fullWidth
                  size="small"
                  placeholder="Matière..."
                  value={searchFilters.subject}
                  onChange={(e) => handleSearchChange('subject', e.target.value)}
                  InputProps={{
                    startAdornment: (
                      <InputAdornment position="start">
                        <Iconify icon="eva:search-fill" sx={{ color: 'text.disabled' }} />
                      </InputAdornment>
                    ),
                    endAdornment: searchFilters.subject && (
                      <InputAdornment position="end">
                        <IconButton size="small" onClick={() => clearColumnFilter('subject')}>
                          <Iconify icon="eva:close-fill" />
                        </IconButton>
                      </InputAdornment>
                    )
                  }}
                />
              </TableCell>
              <TableCell sx={{ width: '12.5%' }}>
                <TextField
                  fullWidth
                  size="small"
                  placeholder="Chapitre..."
                  value={searchFilters.chapter}
                  onChange={(e) => handleSearchChange('chapter', e.target.value)}
                  InputProps={{
                    startAdornment: (
                      <InputAdornment position="start">
                        <Iconify icon="eva:search-fill" sx={{ color: 'text.disabled' }} />
                      </InputAdornment>
                    ),
                    endAdornment: searchFilters.chapter && (
                      <InputAdornment position="end">
                        <IconButton size="small" onClick={() => clearColumnFilter('chapter')}>
                          <Iconify icon="eva:close-fill" />
                        </IconButton>
                      </InputAdornment>
                    )
                  }}
                />
              </TableCell>
              <TableCell sx={{ width: '12.5%' }}>
                <TextField
                  fullWidth
                  size="small"
                  placeholder="Exercice..."
                  value={searchFilters.exercise}
                  onChange={(e) => handleSearchChange('exercise', e.target.value)}
                  InputProps={{
                    startAdornment: (
                      <InputAdornment position="start">
                        <Iconify icon="eva:search-fill" sx={{ color: 'text.disabled' }} />
                      </InputAdornment>
                    ),
                    endAdornment: searchFilters.exercise && (
                      <InputAdornment position="end">
                        <IconButton size="small" onClick={() => clearColumnFilter('exercise')}>
                          <Iconify icon="eva:close-fill" />
                        </IconButton>
                      </InputAdornment>
                    )
                  }}
                />
              </TableCell>
              <TableCell sx={{ width: '12.5%' }}>
                <TextField
                  fullWidth
                  size="small"
                  placeholder="Statut..."
                  value={searchFilters.status}
                  onChange={(e) => handleSearchChange('status', e.target.value)}
                  InputProps={{
                    startAdornment: (
                      <InputAdornment position="start">
                        <Iconify icon="eva:search-fill" sx={{ color: 'text.disabled' }} />
                      </InputAdornment>
                    ),
                    endAdornment: searchFilters.status && (
                      <InputAdornment position="end">
                        <IconButton size="small" onClick={() => clearColumnFilter('status')}>
                          <Iconify icon="eva:close-fill" />
                        </IconButton>
                      </InputAdornment>
                    )
                  }}
                />
              </TableCell>
              <TableCell sx={{ width: '12.5%' }}>
                <IconButton
                  size="small"
                  onClick={clearAllColumnFilters}
                  disabled={!Object.values(searchFilters).some(v => !!v)}
                  sx={{
                    width: '100%',
                    color: 'error.main',
                    '&.Mui-disabled': {
                      color: 'text.disabled',
                    }
                  }}
                >
                  <Iconify icon="eva:trash-2-outline" />
                </IconButton>
              </TableCell>
            </TableRow>
          </Table>
        </TableContainer>

        {/* Corps de la table scrollable */}
        <TableContainer
          sx={{
            position: 'relative',
            height: 'calc(100% - 90px)', // Hauteur ajustée pour tenir compte de l'en-tête et de la ligne de recherche
            overflow: 'auto',
            '&::-webkit-scrollbar': {
              width: '8px',
            },
            '&::-webkit-scrollbar-thumb': {
              backgroundColor: 'rgba(0,0,0,0.2)',
              borderRadius: '4px',
            },
          }}
        >
          <Table
            size={dense ? 'small' : 'medium'}
            sx={{
              minWidth: 960,
              tableLayout: 'fixed', // Force des colonnes égales
              position: 'relative',
            }}
          >
            {/* TableHead invisible pour maintenir l'alignement des colonnes */}
            <TableHeadCustom
              order={order}
              orderBy={orderBy}
              headLabel={TABLE_HEAD}
              onSort={onSort}
              sx={{ visibility: 'collapse', height: 0, padding: 0 }}
            />

            <TableBody>
              {notFound ? (
                <TableNoData notFound={notFound} />
              ) : (
                <>
                  {tableDataFiltered
                    .slice(page * rowsPerPage, page * rowsPerPage + rowsPerPage)
                    .map((row) => (
                      <AIAssistantTableRow
                        key={row.id}
                        row={row}
                        selected={false} // Ajout de la prop manquante
                        onSelectRow={() => dummySelectRow(row.id)} // Ajout de la prop manquante
                        onDeleteRow={() => onDeleteRow(row.id)}
                        onEditRow={() => onEditRow(row.id)}
                        onSettingsRow={() => handleSettings(row.id)}
                        onOpenDeleteConfirm={() => handleOpenDeleteConfirm(row.id)}
                      />
                    ))}
                </>
              )}
            </TableBody>
          </Table>
        </TableContainer>
      </Box>

      <TablePaginationCustom
        count={tableDataFiltered.length}
        page={page}
        rowsPerPage={rowsPerPage}
        onPageChange={onChangePage}
        onRowsPerPageChange={onChangeRowsPerPage}
        //
        dense={dense}
        sx={{
          [`& .${tablePaginationClasses.toolbar}`]: {
            borderTopColor: 'transparent',
          },
        }}
      />

      {/* Boîte de dialogue de confirmation de suppression */}
      <Dialog
        open={deleteDialog.value}
        onClose={deleteDialog.onFalse}
        aria-labelledby="alert-dialog-title"
        aria-describedby="alert-dialog-description"
      >
        <DialogTitle id="alert-dialog-title">
          Supprimer l'assistant
        </DialogTitle>
        <DialogContent>
          <DialogContentText id="alert-dialog-description">
            Êtes-vous sûr de vouloir supprimer cet assistant IA ? Cette action est irréversible.
          </DialogContentText>
        </DialogContent>
        <DialogActions>
          <Button onClick={deleteDialog.onFalse} color="inherit">
            Annuler
          </Button>
          <Button onClick={handleConfirmDelete} color="error" autoFocus>
            Supprimer
          </Button>
        </DialogActions>
      </Dialog>
    </>
  );
}