'use client';

import { useState, useCallback, useEffect } from 'react'; // Add useEffect

import Box from '@mui/material/Box';
import Stack from '@mui/material/Stack';
import Button from '@mui/material/Button';
import Typography from '@mui/material/Typography';
import { useBoolean } from 'src/hooks/use-boolean';
import { DashboardContent } from 'src/shared/layouts/dashboard';
import { Iconify } from 'src/shared/components/iconify';
import { EmptyContent } from 'src/shared/components/empty-content';
import { useTable, getComparator, rowInPage } from 'src/shared/components/table';
import { AIAssistantTable } from './ai-assistant-table';
import { AIAssistantForm } from './ai-assistant-form';
import { _mockAIAssistants } from 'src/shared/_mock/_ai-assistant';
import type { IAIAssistantItem, IAIAssistantTableFilters } from 'src/types/ai-assistant';
import { useRouter } from 'next/navigation'; // Use from navigation instead of router for app directory
import {updateAssistantsWithFakeDates} from "./ai-assistant-service"
import { DatePicker } from '@mui/x-date-pickers/DatePicker';
import { LocalizationProvider } from '@mui/x-date-pickers/LocalizationProvider';
import { AdapterDateFns } from '@mui/x-date-pickers/AdapterDateFns';

export function PersonalizationAiView() {
  const router = useRouter();
  const addDialog = useBoolean();
  const editDialog = useBoolean();
  const [tableData, setTableData] = useState<IAIAssistantItem[]>(_mockAIAssistants);
  const [currentAssistant, setCurrentAssistant] = useState<IAIAssistantItem | null>(null);
  // Initialisez la table avec un tri par défaut sur la date d'ajout (ordre décroissant)
  const table = useTable({ 
    defaultRowsPerPage: 10,
    defaultOrderBy: 'createdAt',    // Définir la colonne par défaut
    defaultOrder: 'desc'            // Trier par ordre décroissant (plus récent en premier)
  });
  // État pour les filtres
  const [filters, setFilters] = useState<IAIAssistantTableFilters>({
    name: '',
    type: [],
    status: '',
    educationLevel: '',
    subject: '',
    chapter: '',
    exercise: '',
  });
  
  // Vérifier si le type "J'apprends" est sélectionné
  const isJapprendsSelected = filters.type.includes('japprends');
  
  const dataFiltered = applyFilter({
    inputData: tableData.map((assistant) => ({
      ...assistant,
      createdAt: assistant.createdAt || "-", // Assurer qu'il y ait toujours une valeur
    })),
    comparator: getComparator(table.order, table.orderBy),
    filters,
  });
  
  useEffect(() => {
    const updatedData = dataFiltered.map(assistant => ({
      ...assistant,
      description: assistant.description || "Description non disponible",
    }));
    localStorage.setItem('assistants', JSON.stringify(updatedData));
  }, [dataFiltered]);
  

  const [isMounted, setIsMounted] = useState(false);
  useEffect(() => {
    // Mettre à jour les dates manquantes pour tous les assistants
    updateAssistantsWithFakeDates();
    
    // Ensuite, récupérer les données mises à jour depuis le localStorage
    const storedAssistants = JSON.parse(localStorage.getItem('assistants') || '[]');
    if (storedAssistants.length > 0) {
      setTableData(storedAssistants);
    } else {
      // Si aucune donnée n'existe dans localStorage, utiliser les données mockées
      setTableData(_mockAIAssistants);
      // Et les sauvegarder immédiatement dans localStorage
      localStorage.setItem('assistants', JSON.stringify(_mockAIAssistants));
    }
  }, []);
  
  
  // Make sure router is only used after component is mounted
  useEffect(() => {
    setIsMounted(true);
  }, []);
  const handleSettingsRow = useCallback((id: string) => {
    if (isMounted) {
      router.push(`/dashboard/personalization-ai/settings/${id}`);
    }
  }, [router, isMounted]);
  const dataInPage = rowInPage(dataFiltered, table.page, table.rowsPerPage);
  const notFound = !dataFiltered.length;

  const handleDeleteItem = useCallback(
    (id: string) => {
      const deleteRow = tableData.filter((row) => row.id !== id);
      setTableData(deleteRow);
      table.onUpdatePageDeleteRow(dataInPage.length);
    },
    [dataInPage.length, table, tableData]
  );

  const handleEditRow = useCallback((id: string) => {
    const assistantToEdit = tableData.find(assistant => assistant.id === id);
    if (assistantToEdit) {
      setCurrentAssistant(assistantToEdit);
      editDialog.onTrue();
    }
  }, [tableData, editDialog]);

  const handleAddAssistant = (newAssistant: IAIAssistantItem) => {
    // Générer un ID pour le nouvel assistant si nécessaire
    if (!newAssistant.id) {
      newAssistant.id = `ai-assistant-${Date.now()}`;
    }
    setTableData([...tableData, newAssistant]);
  };

  const handleUpdateAssistant = (updatedAssistant: IAIAssistantItem) => {
    setTableData(tableData.map(item => 
      item.id === updatedAssistant.id ? updatedAssistant : item
    ));
  };

  // Fonction pour gérer les changements de filtres
  const handleFilterChange = useCallback((filterName: string, value: string | string[]) => {
    setFilters(prevFilters => ({
      ...prevFilters,
      [filterName]: value,
    }));
  }, []);


  return (
    <>
      <DashboardContent maxWidth="xl">
        <Stack direction="row" alignItems="center" justifyContent="space-between">
        <Typography variant="h4" sx={{ textAlign: "center", flexGrow: 1 }}>
          Personnalisation IA
        </Typography>
          <Button
            variant="contained"
            startIcon={<Iconify icon="eva:plus-fill" />}
            onClick={addDialog.onTrue}
          >
            Ajouter un assistant
          </Button>
        </Stack>
        
        {/* Toujours afficher le composant de tableau avec ses filtres */}
        <AIAssistantTable
          table={table}
          dataFiltered={dataFiltered}
          notFound={notFound}
          onDeleteRow={handleDeleteItem}
          onEditRow={handleEditRow}
          onSettingsRow={handleSettingsRow}
          filters={filters}
          isJapprendsSelected={isJapprendsSelected}
          onFilterChange={handleFilterChange}
        />



        {/* Afficher le message de "Aucun résultat" à l'intérieur du tableau plutôt qu'à sa place */}
        {/* La condition notFound est maintenant gérée à l'intérieur du composant AIAssistantTable */}
      </DashboardContent>
      
      {/* Formulaire d'ajout d'assistant */}
      <AIAssistantForm 
        open={addDialog.value} 
        onClose={addDialog.onFalse}
        onSubmit={handleAddAssistant} 
        isEdit={false}
      />
      
      {/* Formulaire de modification d'assistant */}
      <AIAssistantForm
        open={editDialog.value}
        onClose={editDialog.onFalse}
        onSubmit={handleUpdateAssistant}
        onDelete={handleDeleteItem}
        initialData={currentAssistant}
        isEdit={true}
      />
    </>
  );
}

// La fonction applyFilter
type ApplyFilterProps = {
  inputData: IAIAssistantItem[];
  comparator: (a: any, b: any) => number;
  filters: IAIAssistantTableFilters;
};

function applyFilter({ inputData, comparator, filters }: ApplyFilterProps) {
  const { name, type, status, educationLevel, subject, chapter, exercise } = filters;
  
  // Filtrer les données selon les critères
  const filtered = inputData.filter(item => {
    // Filtrer par nom
    if (name && !item.name.toLowerCase().includes(name.toLowerCase())) {
      return false;
    }
    // Filtrer par type
    if (type && type.length > 0 && !type.includes(item.type)) {
      return false;
    }
    // Filtrer par statut
    if (status && item.status !== status) {
      return false;
    }
    // Filtrer par niveau d'éducation
    if (educationLevel && item.educationLevel !== educationLevel) {
      return false;
    }
    // Les filtres suivants ne s'appliquent que si le type "J'apprends" est sélectionné
    if (type && type.includes('japprends')) {
      // Filtrer par matière
      if (subject && (!item.subject || item.subject !== subject)) {
        return false;
      }
      // Filtrer par chapitre - comparaison exacte maintenant que les formats sont harmonisés
      if (chapter && (!item.chapter || item.chapter !== chapter)) {
        return false;
      }
      // Filtrer par exercice - comparaison exacte maintenant que les formats sont harmonisés
      if (exercise && (!item.exercise || item.exercise !== exercise)) {
        return false;
      }
    }
    return true;
  });

  // Trier les données filtrées
  const stabilizedThis = filtered.map((el, index) => [el, index] as const);
  stabilizedThis.sort((a, b) => {
    const order = comparator(a[0], b[0]);
    if (order !== 0) return order;
    return a[1] - b[1];
  });
  
  return stabilizedThis.map((el) => el[0]);
}