'use client';

import type { ISubject } from 'src/contexts/types/common';
import type { IAbonnementItem } from 'src/contexts/types/abonnement';
import type { IAdmin, IChild, IParent, BaseUser } from 'src/contexts/types/user';

import React, { useMemo, useState, useCallback } from 'react';

import Box from '@mui/material/Box';
import Tab from '@mui/material/Tab';
import Card from '@mui/material/Card';
import Grid from '@mui/material/Grid';
import Tabs from '@mui/material/Tabs';
import { Button } from '@mui/material';
import Alert from '@mui/material/Alert';
import Divider from '@mui/material/Divider';
import KeyIcon from '@mui/icons-material/Key';
import EditIcon from '@mui/icons-material/Edit';
import BookIcon from '@mui/icons-material/Book';
import Typography from '@mui/material/Typography';
import PersonIcon from '@mui/icons-material/Person';
import SecurityIcon from '@mui/icons-material/Security';
import CheckCircleIcon from '@mui/icons-material/CheckCircle';
import ShoppingCartIcon from '@mui/icons-material/ShoppingCart';

import { abonnementItems } from 'src/shared/_mock';

import { toast } from 'src/shared/components/snackbar';
import ConditionalComponent from 'src/shared/components/ConditionalComponent/ConditionalComponent';

import TabPanel from './components/TabPanel';
import SecurityTab from './components/SecurityTab';
import PersonalInfoTab from './components/PersonalInfoTab';
import { SubjectsModal } from './components/SubjectsModal';
import AdminRoleSelector from './components/AdminRoleSelector';
import { ChangePlanDialog } from './components/ChangePlanDialog';

// Schéma pour la réinitialisation du mot de passe

// Types additionnels pour les données mockées
interface MockedChildData extends IChild {
  subjects?: ISubject[];
  daily_question_limit?: number;
  subscription?: {
    id: string;
    title: string;
    nbr_subjects: number;
    daily_question_limit: number;
  };
}

interface MockedParentData extends IParent {
  subscription?: {
    id: string;
    title: string;
    nbr_subjects: number;
    daily_question_limit: number;
    interval: string;
  };
}

type ExtendedUserResponse = BaseUser | MockedChildData | MockedParentData | IAdmin;

const AVAILABLE_SUBJECTS: ISubject[] = [
  { id: 'subj1', name: 'Mathématiques', isSelected: false },
  { id: 'subj2', name: 'Français', isSelected: false },
  { id: 'subj3', name: 'Physique-Chimie', isSelected: false },
  { id: 'subj4', name: 'SVT', isSelected: false },
  { id: 'subj5', name: 'Histoire-Géographie', isSelected: false },
  { id: 'subj6', name: 'Anglais', isSelected: false },
  { id: 'subj7', name: 'Informatique', isSelected: false },
  { id: 'subj8', name: 'Français B1', isSelected: false },
];

type Props = {
  currentUser?: BaseUser;
};

export function UserConsulterForm({ currentUser }: Props) {
  const [tabValue, setTabValue] = useState(0);
  const [changePlanDialogOpen, setChangePlanDialogOpen] = useState(false);

  const [currentChildId, setCurrentChildId] = useState<string | null>(null);
  const [subjectsDialogOpen, setSubjectsDialogOpen] = useState(false);
  const [availableSubjects, setAvailableSubjects] = useState<ISubject[]>([]);

  // États pour la gestion des rôles administrateur
  const [selectedAdminRoles, setSelectedAdminRoles] = useState<string[]>([]);
  const [stepErrors, setStepErrors] = useState<{ adminRoles?: string }>({});

  // Mock des données pour les enfants si elles n'existent pas
  const getMockedChildData = useCallback((user: ExtendedUserResponse): MockedChildData | null => {
    if (user.userType !== 'CHILD') return null;

    const childUser = user as IChild;
    return {
      ...childUser,
      subjects: [
        { id: 'subj1', name: 'Mathématiques', isSelected: true },
        { id: 'subj2', name: 'Français', isSelected: true },
        { id: 'subj3', name: 'Physique-Chimie', isSelected: false },
      ],
      daily_question_limit: 12,
      subscription: {
        id: 'basic',
        title: 'Plan Basique',
        nbr_subjects: 5,
        daily_question_limit: 10,
      },
    };
  }, []);

  // Mock des données pour les parents si elles n'existent pas
  const getMockedParentData = useCallback((user: ExtendedUserResponse): MockedParentData | null => {
    if (user.userType !== 'PARENT') return null;

    const parentUser = user as IParent;
    return {
      ...parentUser,
      subscription: {
        id: 'premium',
        title: 'Plan Premium',
        nbr_subjects: 10,
        daily_question_limit: 20,
        interval: 'monthly',
      },
    };
  }, []);

  // Convertir les matières de l'enfant en format attendu par le composant
  const childSubjects = useMemo(() => {
    if (currentUser?.userType === 'CHILD') {
      const mockedChild = getMockedChildData(currentUser);
      if (mockedChild?.subjects) {
        return mockedChild.subjects.map((subject) => ({
          ...subject,
          isSelected: true,
        }));
      }
    }
    return [];
  }, [currentUser, getMockedChildData]);

  // Calculer la liste complète des matières disponibles
  const fullSubjectsList = useMemo(() => {
    if (currentUser?.userType === 'CHILD') {
      const selectedSubjects = childSubjects;
      const selectedIds = selectedSubjects.map((s) => s.id);
      return [
        ...selectedSubjects,
        ...AVAILABLE_SUBJECTS.filter((s) => !selectedIds.includes(s.id)),
      ];
    }
    return AVAILABLE_SUBJECTS;
  }, [currentUser, childSubjects]);

  // Obtenir les données mockées d'abonnement pour les parents
  const parentSubscription = useMemo(() => {
    if (currentUser?.userType === 'PARENT') {
      const mockedParent = getMockedParentData(currentUser);
      return mockedParent?.subscription;
    }
    return null;
  }, [currentUser, getMockedParentData]);

  const handleTabChange = (event: React.SyntheticEvent, newValue: number) => {
    setTabValue(newValue);
  };

  // Fonction pour ouvrir le dialogue de changement de plan
  const openChangePlanDialog = useCallback(() => {
    setChangePlanDialogOpen(true);
  }, []);

  // Ouvrir le modal pour gérer les matières d'un enfant spécifique
  const openChildSubjectsModal = useCallback(() => {
    if (currentUser?.userType !== 'CHILD') return;

    setAvailableSubjects(fullSubjectsList);
    setSubjectsDialogOpen(true);
  }, [currentUser, fullSubjectsList]);

  // Basculer la sélection d'une matière
  const toggleSubject = useCallback(
    (subjectId: string) => {
      if (currentUser?.userType !== 'CHILD') return;

      setAvailableSubjects((currentSubjects) => {
        const subject = currentSubjects.find((s) => s.id === subjectId);
        if (!subject) return currentSubjects;

        // Si la matière est déjà sélectionnée, on peut toujours la désélectionner
        if (subject.isSelected) {
          return currentSubjects.map((s) => (s.id === subjectId ? { ...s, isSelected: false } : s));
        }

        // Vérifier le nombre de matières déjà sélectionnées
        const selectedCount = currentSubjects.filter((s) => s.isSelected).length;
        const dailyLimit = 5;

        if (selectedCount >= dailyLimit) {
          toast.error(`Cet enfant ne peut pas sélectionner plus de ${dailyLimit} matières.`);
          return currentSubjects;
        }

        return currentSubjects.map((s) => (s.id === subjectId ? { ...s, isSelected: true } : s));
      });
    },
    [currentUser]
  );

  // Sauvegarder les matières sélectionnées pour l'enfant actuel
  const saveSubjectsChanges = useCallback(() => {
    if (currentUser?.userType !== 'CHILD') return;

    // Stocker pour l'enfant directement
    console.log(
      "Matières sauvegardées pour l'enfant:",
      availableSubjects.filter((s) => s.isSelected)
    );
    toast.success(`Les matières ont été assignées avec succès.`);
    setSubjectsDialogOpen(false);
  }, [availableSubjects, currentUser]);

  // Fonction pour changer de plan d'abonnement
  const handleChangePlan = useCallback((planId: string, interval: string) => {
    // Trouver le nouveau plan
    const newPlan = abonnementItems.find((plan) => plan.id === planId) as IAbonnementItem;
    if (!newPlan) {
      toast.error("Ce plan d'abonnement n'existe pas.");
      return;
    }

    setChangePlanDialogOpen(false);
    toast.success(`Abonnement souscrit avec succès: ${newPlan.title}`);
  }, []);

  // Fonction pour gérer l'édition des rôles administrateur
  const handleAdminRolesEdit = useCallback(() => {
    if (selectedAdminRoles.length === 0) {
      setStepErrors({ adminRoles: 'Veuillez sélectionner au moins un rôle' });
      return;
    }

    console.log('Rôles administrateur sauvegardés:', selectedAdminRoles);
    toast.success('Les rôles administrateur ont été mis à jour avec succès');
    setStepErrors({});
  }, [selectedAdminRoles]);

  const avatarSrc = currentUser?.photo || undefined;

  // Déterminer l'index correct pour chaque onglet
  const getTabIndex = () => {
    let index = 0;
    const tabs = {
      personalInfo: index,
      subscription: currentUser?.userType === 'PARENT' ? (index += 1) : -1,
      subjects: currentUser?.userType === 'CHILD' ? (index += 1) : -1,
      adminRoles: currentUser?.userType === 'ADMIN' ? (index += 1) : -1,
      security: (index += 1),
    };
    return tabs;
  };

  const tabIndexes = getTabIndex();

  return (
    <>
      <Tabs
        value={tabValue}
        onChange={handleTabChange}
        aria-label="user management tabs"
        sx={{ borderBottom: 1, borderColor: 'divider', mb: 2 }}
      >
        <Tab icon={<PersonIcon />} label="Informations personnelles" />
        {currentUser?.userType === 'PARENT' && (
          <Tab icon={<BookIcon />} label="Abonnement et matières" />
        )}
        {currentUser?.userType === 'CHILD' && <Tab icon={<BookIcon />} label="Matières" />}
        {currentUser?.userType === 'ADMIN' && (
          <Tab icon={<SecurityIcon />} label="Rôles administrateur" />
        )}
        <Tab icon={<KeyIcon />} label="Sécurité du compte" />
      </Tabs>

      {/* Informations Personnelles */}
      <TabPanel value={tabValue} index={tabIndexes.personalInfo}>
        <PersonalInfoTab currentUser={currentUser} />
      </TabPanel>

      {/* Tab pour Abonnement et Matières (Parent) */}
      <ConditionalComponent isValid={currentUser?.userType === 'PARENT'}>
        <TabPanel value={tabValue} index={tabIndexes.subscription}>
          <Card sx={{ p: 3 }}>
            <Typography variant="h6" sx={{ mb: 2 }}>
              Aucun abonnement actif
            </Typography>
            <Divider sx={{ mb: 3 }} />
            <Alert severity="info" sx={{ mb: 3 }}>
              Vous n&apos;avez pas d&apos;abonnement actif. Choisissez un abonnement pour accéder
              aux fonctionnalités et assigner des matières à vos enfants.
            </Alert>
            <Box sx={{ display: 'flex', justifyContent: 'center', mt: 2 }}>
              <Button
                variant="contained"
                color="primary"
                size="large"
                startIcon={<ShoppingCartIcon />}
                onClick={openChangePlanDialog}
              >
                Choisir un abonnement
              </Button>
            </Box>
          </Card>
        </TabPanel>
      </ConditionalComponent>

      {/* Tab pour Matières (Enfant) */}
      <ConditionalComponent isValid={currentUser?.userType === 'CHILD'}>
        <TabPanel value={tabValue} index={tabIndexes.subjects}>
          <Grid container spacing={3}>
            {/* Section Matières */}
            <Grid item xs={12}>
              <Card sx={{ p: 3 }}>
                <Typography variant="h6" sx={{ mb: 2 }}>
                  <BookIcon sx={{ mr: 1, verticalAlign: 'middle' }} />
                  Matières assignées
                </Typography>
                <Divider sx={{ mb: 3 }} />

                {/* Affichage des matières */}
                <Grid container spacing={2} sx={{ mb: 3 }}>
                  {childSubjects.length > 0 ? (
                    childSubjects.map((subject) => (
                      <Grid item xs={12} sm={6} md={4} key={subject.id}>
                        <Card
                          sx={{
                            p: 2,
                            bgcolor: 'primary.lighter',
                            border: '1px solid',
                            borderColor: 'primary.main',
                          }}
                        >
                          <Box
                            sx={{
                              display: 'flex',
                              justifyContent: 'space-between',
                              alignItems: 'center',
                            }}
                          >
                            <Typography variant="subtitle2">{subject.name}</Typography>
                            <CheckCircleIcon color="primary" />
                          </Box>
                        </Card>
                      </Grid>
                    ))
                  ) : (
                    <Grid item xs={12}>
                      <Alert severity="info">
                        Aucune matière n&apos;est actuellement assignée à cet enfant.
                      </Alert>
                    </Grid>
                  )}
                </Grid>

                {/* Limite quotidienne */}
                <Typography variant="body2" color="text.secondary" sx={{ mb: 2 }}>
                  Limite quotidienne: {5} questions par jour
                </Typography>

                {/* Boutons d'action */}
                <Box sx={{ mt: 3, display: 'flex', justifyContent: 'end' }}>
                  <Button
                    startIcon={<EditIcon />}
                    variant="contained"
                    onClick={openChildSubjectsModal}
                  >
                    Modifier les matières
                  </Button>
                </Box>
              </Card>
            </Grid>
          </Grid>
        </TabPanel>
      </ConditionalComponent>

      {/* Tab pour les rôles administrateur */}
      <ConditionalComponent isValid={currentUser?.userType === 'ADMIN'}>
        <TabPanel value={tabValue} index={tabIndexes.adminRoles}>
          <AdminRoleSelector currentUser={currentUser} />
        </TabPanel>
      </ConditionalComponent>

      {/* Tab pour Sécurité du compte */}
      <TabPanel value={tabValue} index={tabIndexes.security}>
        <SecurityTab currentUser={currentUser} />
      </TabPanel>

      {/* Modal pour la gestion des matières */}
      <SubjectsModal
        open={subjectsDialogOpen}
        onClose={() => setSubjectsDialogOpen(false)}
        subjects={availableSubjects}
        onToggleSubject={toggleSubject}
        onSave={saveSubjectsChanges}
      />

      {/* Dialog de changement de plan */}
      <ChangePlanDialog
        open={changePlanDialogOpen}
        onClose={() => setChangePlanDialogOpen(false)}
        currentSubscription={undefined}
        availablePlans={abonnementItems}
        onChangePlan={handleChangePlan}
      />
    </>
  );
}
