'use client';

import type { IUserResponse } from 'src/contexts/types/user';

import dayjs from 'dayjs';
import { useState } from 'react';
import { FontAwesomeIcon } from '@fortawesome/react-fontawesome';
import {
  faEye,
  faBan,
  faUndo,
  faTrash,
  faPause,
  faUnlock,
  faEllipsisV,
  faUserCheck,
  faPenToSquare,
} from '@fortawesome/free-solid-svg-icons';

import Menu from '@mui/material/Menu';
import Button from '@mui/material/Button';
import MenuItem from '@mui/material/MenuItem';
import TableRow from '@mui/material/TableRow';
import Checkbox from '@mui/material/Checkbox';
import TableCell from '@mui/material/TableCell';
import TextField from '@mui/material/TextField';
import IconButton from '@mui/material/IconButton';
import InputLabel from '@mui/material/InputLabel';
import FormControl from '@mui/material/FormControl';
import ListItemIcon from '@mui/material/ListItemIcon';
import ListItemText from '@mui/material/ListItemText';
import { Box, Select, Tooltip, CircularProgress } from '@mui/material';

import { paths } from 'src/routes/paths';
import { useRouter } from 'src/routes/hooks';

import { useBoolean } from 'src/hooks';
import { useUserStore } from 'src/shared/api/stores/userStore';
import { getStatusLabel, mapUserTypeToRole } from 'src/contexts/types/user';

import { Label } from 'src/shared/components/label';
import { toast } from 'src/shared/components/snackbar';
import { ConfirmDialog } from 'src/shared/components/custom-dialog/confirm-dialog';

const SUSPEND_DURATIONS = [7, 14, 30];

type Props = {
  row: IUserResponse;
  selected: boolean;
  onSelectRow: () => void;
  statusFilter?: string;
  columns: Array<{ id: string; label: string; width?: string | number }>;
};

interface DeletedParent {
  id: string;
  lastName: string;
  firstName: string;
}

interface SuspensionRequest {
  reason: string;
  suspensionEnd: string;
}

export function UserTableRow({
  row,
  selected,
  onSelectRow,
  statusFilter = 'Tous',
  columns,
}: Props) {
  const [anchorEl, setAnchorEl] = useState<null | HTMLElement>(null);
  const menuOpen = Boolean(anchorEl);
  const router = useRouter();

  const confirmDelete = useBoolean();
  const confirmBlock = useBoolean();
  const confirmSuspend = useBoolean();
  const confirmReactivate = useBoolean();

  const [isDeleting, setIsDeleting] = useState(false);
  const [isBlocking, setIsBlocking] = useState(false);
  const [isSuspending, setIsSuspending] = useState(false);
  const [isReactivating, setIsReactivating] = useState(false);
  const [isBlockingChildren, setIsBlockingChildren] = useState(false);
  const [isSuspendingChildren, setIsSuspendingChildren] = useState(false);
  const [isReactivatingChildren, setIsReactivatingChildren] = useState(false);

  const [deletedParentInfo, setDeletedParentInfo] = useState<DeletedParent | null>(null);
  const [parentInfo, setParentInfo] = useState<DeletedParent | null>(null);

  const confirmDeleteChildren = useBoolean();
  const confirmBlockChildren = useBoolean();
  const confirmSuspendChildren = useBoolean();
  const confirmReactivateChildren = useBoolean();
  const {
    deleteAdmin,
    deleteChild,
    deleteParent,
    deleteChildrenByParent,

    suspendAdmin,
    suspendChild,
    suspendParent,
    suspendChildrenByParent,

    blockAdmin,
    blockChild,
    blockParent,
    blockChildrenByParent,

    reactivateAdmin,
    reactivateChild,
    reactivateParent,
    reactivateChildrenByParent,
    loading,
  } = useUserStore();

  const [blockReason, setBlockReason] = useState('');
  const [suspendReason, setSuspendReason] = useState('');
  const [suspendDuration, setSuspendDuration] = useState(SUSPEND_DURATIONS[0]);

  const handleOpenMenu = (event: React.MouseEvent<HTMLElement>) => {
    setAnchorEl(event.currentTarget);
  };

  const handleCloseMenu = () => {
    setAnchorEl(null);
  };

  const calculateSuspensionEndDate = (days: number): string =>
    dayjs().add(days, 'day').toISOString();

  const getUserType = (): string => row.role?.toLowerCase() || row.userType?.toLowerCase() || '';

  const handleDeleteUser = async () => {
    setIsDeleting(true);

    try {
      const userType = getUserType();

      switch (userType) {
        case 'parent':
          setDeletedParentInfo({ id: row.id, firstName: row.firstName, lastName: row.lastName });
          confirmDelete.onFalse();
          setTimeout(() => {
            confirmDeleteChildren.onTrue();
          }, 100);
          break;

        case 'child':
        case 'enfant':
          await deleteChild(row.id);
          toast.success(`L'enfant ${row.firstName} ${row.lastName} a été supprimé avec succès.`);
          confirmDelete.onFalse();
          break;

        case 'admin':
        case 'administrateur':
          await deleteAdmin(row.id);
          toast.success(
            `L'administrateur ${row.firstName} ${row.lastName} a été supprimé avec succès.`
          );
          confirmDelete.onFalse();
          break;

        default:
          throw new Error("Type d'utilisateur non reconnu");
      }
    } catch (error: any) {
      console.error('Erreur lors de la suppression:', error);
      toast.error(
        `Erreur lors de la suppression: ${error.message || 'Une erreur inconnue est survenue'}`
      );
    } finally {
      setIsDeleting(false);
    }
  };

  const handleBlockUser = async () => {
    setIsBlocking(true);

    try {
      const userType = getUserType();

      switch (userType) {
        case 'parent':
          setParentInfo({ id: row.id, firstName: row.firstName, lastName: row.lastName });
          confirmBlock.onFalse();
          setBlockReason('');
          setTimeout(() => {
            confirmBlockChildren.onTrue();
          }, 100);
          break;
        case 'child':
        case 'enfant':
          await blockChild(row.id, blockReason);
          toast.success(`L'enfant ${row.firstName} ${row.lastName} a été bloqué avec succès.`);
          confirmBlock.onFalse();
          setBlockReason('');
          break;
        case 'admin':
        case 'administrateur':
          await blockAdmin(row.id, blockReason);
          toast.success(
            `L'administrateur ${row.firstName} ${row.lastName} a été bloqué avec succès.`
          );
          confirmBlock.onFalse();
          setBlockReason('');
          break;
        default:
          throw new Error("Type d'utilisateur non reconnu");
      }
    } catch (error: any) {
      console.error('Erreur lors du blocage:', error);
      toast.error(`Erreur lors du blocage: ${error.message || 'Une erreur inconnue est survenue'}`);
    } finally {
      setIsBlocking(false);
    }
  };

  const handleSuspendUser = async () => {
    if (!suspendReason.trim() || !suspendDuration) {
      toast.error('Veuillez remplir tous les champs requis');
      return;
    }

    setIsSuspending(true);

    try {
      const userType = getUserType();
      const suspensionEndDate = calculateSuspensionEndDate(suspendDuration);
      const suspensionRequest: SuspensionRequest = {
        reason: suspendReason,
        suspensionEnd: suspensionEndDate,
      };

      switch (userType) {
        case 'parent':
          setParentInfo({ id: row.id, firstName: row.firstName, lastName: row.lastName });
          confirmSuspend.onFalse();
          setSuspendReason('');
          setSuspendDuration(SUSPEND_DURATIONS[0]);
          confirmSuspendChildren.onTrue();
          break;
        case 'child':
        case 'enfant':
          await suspendChild(row.id, suspensionRequest);
          toast.success(
            `L'enfant ${row.firstName} ${row.lastName} a été suspendu pour ${suspendDuration} jours.`
          );
          confirmSuspend.onFalse();
          setSuspendReason('');
          setSuspendDuration(SUSPEND_DURATIONS[0]);
          break;
        case 'admin':
        case 'administrateur':
          await suspendAdmin(row.id, suspensionRequest);
          toast.success(
            `L'administrateur ${row.firstName} ${row.lastName} a été suspendu pour ${suspendDuration} jours.`
          );
          confirmSuspend.onFalse();
          setSuspendReason('');
          setSuspendDuration(SUSPEND_DURATIONS[0]);
          break;
        default:
          throw new Error("Type d'utilisateur non reconnu");
      }
    } catch (error: any) {
      console.error('Erreur lors de la suspension:', error);
      toast.error(
        `Erreur lors de la suspension: ${error.message || 'Une erreur inconnue est survenue'}`
      );
    } finally {
      setIsSuspending(false);
    }
  };

  const handleReactivateUser = async () => {
    setIsReactivating(true);

    try {
      const userType = getUserType();

      switch (userType) {
        case 'parent':
          setParentInfo({ id: row.id, firstName: row.firstName, lastName: row.lastName });
          confirmReactivate.onFalse();
          confirmReactivateChildren.onTrue();
          break;
        case 'child':
        case 'enfant':
          await reactivateChild(row.id);
          toast.success(`L'enfant ${row.firstName} ${row.lastName} a été réactivé avec succès.`);
          confirmReactivate.onFalse();
          break;
        case 'admin':
        case 'administrateur':
          await reactivateAdmin(row.id);
          toast.success(
            `L'administrateur ${row.firstName} ${row.lastName} a été réactivé avec succès.`
          );
          confirmReactivate.onFalse();
          break;
        default:
          throw new Error("Type d'utilisateur non reconnu");
      }
    } catch (error: any) {
      console.error('Erreur lors de la réactivation:', error);
      toast.error(
        `Erreur lors de la réactivation: ${error.message || 'Une erreur inconnue est survenue'}`
      );
    } finally {
      setIsReactivating(false);
    }
  };

  const handleDeleteChildren = async () => {
    if (!deletedParentInfo) return;

    setIsDeleting(true);
    try {
      await deleteParent(deletedParentInfo.id);
      toast.success(
        `Le parent ${deletedParentInfo.firstName} ${deletedParentInfo.lastName} a été supprimé avec succès.`
      );

      await deleteChildrenByParent(deletedParentInfo.id);
      toast.success(
        `Les enfants de ${deletedParentInfo.firstName} ${deletedParentInfo.lastName} ont été supprimés avec succès.`
      );
    } catch (error: any) {
      console.error('Erreur lors de la suppression:', error);
      toast.error(
        `Erreur lors de la suppression: ${error.message || 'Une erreur inconnue est survenue'}`
      );
    } finally {
      setIsDeleting(false);
      confirmDeleteChildren.onFalse();
      setDeletedParentInfo(null);
    }
  };

  const handleCancelDeleteChildren = async () => {
    if (!deletedParentInfo) return;

    setIsDeleting(true);
    try {
      await deleteParent(deletedParentInfo.id);
      toast.success(
        `Le parent ${deletedParentInfo.firstName} ${deletedParentInfo.lastName} a été supprimé avec succès.`
      );
    } catch (error: any) {
      console.error('Erreur lors de la suppression du parent:', error);
      toast.error(
        `Erreur lors de la suppression du parent: ${error.message || 'Une erreur inconnue est survenue'}`
      );
    } finally {
      setIsDeleting(false);
      confirmDeleteChildren.onFalse();
      setDeletedParentInfo(null);
    }
  };
  const handleBlockChildren = async () => {
    if (!parentInfo) return;

    setIsBlockingChildren(true);
    try {
      await blockParent(parentInfo.id, blockReason);
      toast.success(
        `Le parent ${parentInfo.firstName} ${parentInfo.lastName} a été bloqué avec succès.`
      );

      await blockChildrenByParent(parentInfo.id);
      toast.success(
        `Les enfants de ${parentInfo.firstName} ${parentInfo.lastName} ont été bloqués avec succès.`
      );
    } catch (error: any) {
      console.error('Erreur lors du blocage:', error);
      toast.error(`Erreur lors du blocage: ${error.message || 'Une erreur inconnue est survenue'}`);
    } finally {
      setIsBlockingChildren(false);
      confirmBlockChildren.onFalse();
      setParentInfo(null);
    }
  };

  const handleCancelBlockChildren = async () => {
    if (!parentInfo) return;

    setIsBlockingChildren(true);
    try {
      await blockParent(parentInfo.id, blockReason);
      toast.success(
        `Le parent ${parentInfo.firstName} ${parentInfo.lastName} a été bloqué avec succès.`
      );
    } catch (error: any) {
      console.error('Erreur lors du blocage du parent:', error);
      toast.error(
        `Erreur lors du blocage du parent: ${error.message || 'Une erreur inconnue est survenue'}`
      );
    } finally {
      setIsBlockingChildren(false);
      confirmBlockChildren.onFalse();
      setParentInfo(null);
    }
  };
  const handleSuspendChildren = async () => {
    if (!parentInfo) return;

    setIsSuspendingChildren(true);
    try {
      const suspensionEndDate = calculateSuspensionEndDate(suspendDuration || SUSPEND_DURATIONS[0]);
      const suspensionRequest: SuspensionRequest = {
        reason: suspendReason,
        suspensionEnd: suspensionEndDate,
      };

      await suspendParent(parentInfo.id, suspensionRequest);
      toast.success(
        `Le parent ${parentInfo.firstName} ${parentInfo.lastName} a été suspendu pour ${suspendDuration} jours.`
      );

      await suspendChildrenByParent(parentInfo.id, suspensionEndDate);
      toast.success(
        `Les enfants de ${parentInfo.firstName} ${parentInfo.lastName} ont été suspendus avec succès.`
      );
    } catch (error: any) {
      console.error('Erreur lors de la suspension:', error);
      toast.error(
        `Erreur lors de la suspension: ${error.message || 'Une erreur inconnue est survenue'}`
      );
    } finally {
      setIsSuspendingChildren(false);
      confirmSuspendChildren.onFalse();
      setParentInfo(null);
    }
  };

  const handleCancelSuspendChildren = async () => {
    if (!parentInfo) return;

    setIsSuspendingChildren(true);
    try {
      const suspensionEndDate = calculateSuspensionEndDate(suspendDuration || SUSPEND_DURATIONS[0]);
      const suspensionRequest: SuspensionRequest = {
        reason: suspendReason,
        suspensionEnd: suspensionEndDate,
      };

      await suspendParent(parentInfo.id, suspensionRequest);
      toast.success(
        `Le parent ${parentInfo.firstName} ${parentInfo.lastName} a été suspendu pour ${suspendDuration} jours.`
      );
    } catch (error: any) {
      console.error('Erreur lors de la suspension du parent:', error);
      toast.error(
        `Erreur lors de la suspension du parent: ${error.message || 'Une erreur inconnue est survenue'}`
      );
    } finally {
      setIsSuspendingChildren(false);
      confirmSuspendChildren.onFalse();
      setParentInfo(null);
    }
  };

  const handleReactivateChildren = async () => {
    if (!parentInfo) return;

    setIsReactivatingChildren(true);
    try {
      await reactivateParent(parentInfo.id);
      toast.success(
        `Le parent ${parentInfo.firstName} ${parentInfo.lastName} a été réactivé avec succès.`
      );

      await reactivateChildrenByParent(parentInfo.id);
      toast.success(
        `Les enfants de ${parentInfo.firstName} ${parentInfo.lastName} ont été réactivés avec succès.`
      );
    } catch (error: any) {
      console.error('Erreur lors de la réactivation:', error);
      toast.error(
        `Erreur lors de la réactivation: ${error.message || 'Une erreur inconnue est survenue'}`
      );
    } finally {
      setIsReactivatingChildren(false);
      confirmReactivateChildren.onFalse();
      setParentInfo(null);
    }
  };

  const handleCancelReactivateChildren = async () => {
    if (!parentInfo) return;

    setIsReactivatingChildren(true);
    try {
      await reactivateParent(parentInfo.id);
      toast.success(
        `Le parent ${parentInfo.firstName} ${parentInfo.lastName} a été réactivé avec succès.`
      );
    } catch (error: any) {
      console.error('Erreur lors de la réactivation du parent:', error);
      toast.error(
        `Erreur lors de la réactivation du parent: ${error.message || 'Une erreur inconnue est survenue'}`
      );
    } finally {
      setIsReactivatingChildren(false);
      confirmReactivateChildren.onFalse();
      setParentInfo(null);
    }
  };

  const renderCellContent = (colId: string) => {
    switch (colId) {
      case 'select':
        return (
          <Checkbox
            checked={selected}
            onClick={(e) => {
              e.stopPropagation();
              onSelectRow();
            }}
            inputProps={{
              id: `row-checkbox-${row.id}`,
              'aria-label': 'row checkbox',
            }}
          />
        );
      case 'name':
        return `${row.firstName} ${row.lastName}`;
      case 'email': {
        const fullEmail = row.email;
        return (
          <Tooltip title={fullEmail}>
            <Box
              sx={{
                overflow: 'hidden',
                textOverflow: 'ellipsis',
                whiteSpace: 'nowrap',
                maxWidth: '100%',
              }}
            >
              {fullEmail}
            </Box>
          </Tooltip>
        );
      }
      case 'role':
        return mapUserTypeToRole(row.role);
      case 'statut':
        return (
          <Label
            variant="soft"
            sx={{
              ...(row.status === 'ACTIVE' && {
                bgcolor: 'rgb(186, 248, 193)',
                color: '#22bb33',
              }),
              ...(row.status === 'DELETED' && {
                bgcolor: 'rgba(244, 67, 54, 0.1)',
                color: '#F44336',
              }),
              ...(row.status === 'BLOCKED' && {
                bgcolor: 'rgba(33, 33, 33, 0.1)',
                color: '#212121',
              }),
              ...(row.status === 'SUSPENDED' && {
                bgcolor: 'rgba(255, 152, 0, 0.1)',
                color: '#FF9800',
              }),
              ...(!['ACTIVE', 'DELETED', 'BLOCKED', 'SUSPENDED'].includes(row.status) && {
                bgcolor: 'rgba(145, 158, 171, 0.16)',
                color: 'text.secondary',
              }),
            }}
          >
            {getStatusLabel(row.status)}
          </Label>
        );
      case 'createdAt':
        return dayjs(row.createdAt).format('DD/MM/YYYY');
      case 'lastLogin':
        return row.lastLogin ? dayjs(row.lastLogin).format('DD/MM/YYYY') : '—';
      case 'actions':
        return (
          <>
            <IconButton onClick={handleOpenMenu} disabled={loading}>
              <FontAwesomeIcon icon={faEllipsisV} />
            </IconButton>
            <Menu
              open={menuOpen}
              anchorEl={anchorEl}
              onClose={handleCloseMenu}
              PaperProps={{ sx: { width: 200 } }}
            >
              {row.status === 'ACTIVE' ? (
                <>
                  <MenuItem
                    onClick={() => {
                      handleCloseMenu();
                      router.push(paths.dashboard.user.consulter(row.id, row.userType));
                    }}
                  >
                    <ListItemIcon>
                      <FontAwesomeIcon icon={faEye} color="#2196F3" />
                    </ListItemIcon>
                    <ListItemText primary="Voir détails" sx={{ color: '#2196F3' }} />
                  </MenuItem>

                  <MenuItem
                    onClick={() => {
                      handleCloseMenu();
                      confirmDelete.onTrue();
                    }}
                  >
                    <ListItemIcon>
                      <FontAwesomeIcon icon={faTrash} color="#F44336" />
                    </ListItemIcon>
                    <ListItemText primary="Supprimer" sx={{ color: '#F44336' }} />
                  </MenuItem>
                  <MenuItem
                    onClick={() => {
                      handleCloseMenu();
                      confirmBlock.onTrue();
                    }}
                  >
                    <ListItemIcon>
                      <FontAwesomeIcon icon={faBan} color="#212121" />
                    </ListItemIcon>
                    <ListItemText primary="Bloquer" sx={{ color: '#212121' }} />
                  </MenuItem>
                  <MenuItem
                    onClick={() => {
                      handleCloseMenu();
                      confirmSuspend.onTrue();
                    }}
                  >
                    <ListItemIcon>
                      <FontAwesomeIcon icon={faPause} color="#FF9800" />
                    </ListItemIcon>
                    <ListItemText primary="Suspendre" sx={{ color: '#FF9800' }} />
                  </MenuItem>
                </>
              ) : row.status === 'DELETED' ? (
                <MenuItem
                  onClick={() => {
                    handleCloseMenu();
                    router.push(paths.dashboard.user.consulter(row.id, row.userType));
                  }}
                >
                  <ListItemIcon>
                    <FontAwesomeIcon icon={faEye} color="#2196F3" />
                  </ListItemIcon>
                  <ListItemText primary="Voir détails" sx={{ color: '#2196F3' }} />
                </MenuItem>
              ) : row.status === 'BLOCKED' ? (
                <>
                  <MenuItem
                    onClick={() => {
                      handleCloseMenu();
                      router.push(paths.dashboard.user.consulter(row.id, row.userType));
                    }}
                  >
                    <ListItemIcon>
                      <FontAwesomeIcon icon={faEye} color="#2196F3" />
                    </ListItemIcon>
                    <ListItemText primary="Voir détails" sx={{ color: '#2196F3' }} />
                  </MenuItem>
                  <MenuItem
                    onClick={() => {
                      handleCloseMenu();
                      confirmReactivate.onTrue();
                    }}
                  >
                    <ListItemIcon>
                      <FontAwesomeIcon icon={faUnlock} color="#22bb33" />
                    </ListItemIcon>
                    <ListItemText primary="Débloquer" sx={{ color: '#22bb33' }} />
                  </MenuItem>
                </>
              ) : row.status === 'SUSPENDED' ? (
                <>
                  <MenuItem
                    onClick={() => {
                      handleCloseMenu();
                      router.push(paths.dashboard.user.consulter(row.id, row.userType));
                    }}
                  >
                    <ListItemIcon>
                      <FontAwesomeIcon icon={faEye} color="#2196F3" />
                    </ListItemIcon>
                    <ListItemText primary="Voir détails" sx={{ color: '#2196F3' }} />
                  </MenuItem>
                  <MenuItem
                    onClick={() => {
                      handleCloseMenu();
                      confirmReactivate.onTrue();
                    }}
                  >
                    <ListItemIcon>
                      <FontAwesomeIcon icon={faUserCheck} color="#22bb33" />
                    </ListItemIcon>
                    <ListItemText primary="Réactiver" sx={{ color: '#22bb33' }} />
                  </MenuItem>
                </>
              ) : null}
            </Menu>
          </>
        );
      default:
        return (row as any)[colId];
    }
  };

  return (
    <>
      <TableRow hover selected={selected} tabIndex={-1}>
        {columns.map((col) => (
          <TableCell key={col.id} sx={{ width: col.width }}>
            {renderCellContent(col.id)}
          </TableCell>
        ))}
      </TableRow>

      {/* Dialog de confirmation de suppression */}
      <ConfirmDialog
        open={confirmDelete.value}
        onClose={confirmDelete.onFalse}
        title={`Supprimer l'utilisateur : ${row.firstName} ${row.lastName}`}
        content="Êtes-vous sûr de vouloir supprimer cet utilisateur ?"
        action={
          <Button
            variant="contained"
            color="error"
            onClick={handleDeleteUser}
            disabled={isDeleting || loading}
            startIcon={isDeleting ? <CircularProgress size={20} /> : null}
          >
            {isDeleting ? 'Suppression...' : 'Supprimer'}
          </Button>
        }
      />

      {/* Dialog de confirmation de suppression des enfants */}
      <ConfirmDialog
        open={confirmDeleteChildren.value}
        onClose={handleCancelDeleteChildren}
        title="Supprimer les enfants du parent ?"
        content={
          deletedParentInfo
            ? `Le parent ${deletedParentInfo.firstName} ${deletedParentInfo.lastName} a été supprimé. Souhaitez-vous également supprimer tous ses enfants ?`
            : ''
        }
        action={
          <Button
            variant="contained"
            color="error"
            onClick={handleDeleteChildren}
            disabled={isDeleting || loading}
            startIcon={isDeleting ? <CircularProgress size={20} /> : null}
          >
            {isDeleting ? 'Suppression...' : 'Oui'}
          </Button>
        }
      />

      {/* Dialog de blocage */}
      <ConfirmDialog
        open={confirmBlock.value}
        onClose={() => {
          confirmBlock.onFalse();
          setBlockReason('');
        }}
        title={`Bloquer l'utilisateur : ${row.firstName} ${row.lastName}`}
        content={
          <TextField
            fullWidth
            label="Motif (optionnel)"
            value={blockReason}
            onChange={(e) => setBlockReason(e.target.value)}
            sx={{ mt: 2 }}
          />
        }
        action={
          <Button
            variant="contained"
            color="error"
            onClick={handleBlockUser}
            disabled={isBlocking || loading}
            startIcon={isBlocking ? <CircularProgress size={20} /> : null}
          >
            {isBlocking ? 'Blocage...' : 'Bloquer'}
          </Button>
        }
      />

      {/* Dialog de suspension */}
      <ConfirmDialog
        open={confirmSuspend.value}
        onClose={() => {
          confirmSuspend.onFalse();
          setSuspendReason('');
          setSuspendDuration(SUSPEND_DURATIONS[0]);
        }}
        title={`Suspendre l'utilisateur : ${row.firstName} ${row.lastName}`}
        content={
          <>
            <TextField
              fullWidth
              label="Motif"
              required
              value={suspendReason}
              onChange={(e) => setSuspendReason(e.target.value)}
              sx={{ mt: 2 }}
            />
            <FormControl fullWidth sx={{ mt: 2 }}>
              <InputLabel id="suspend-duration-label">Durée *</InputLabel>
              <Select
                labelId="suspend-duration-label"
                label="Durée *"
                value={suspendDuration}
                onChange={(e) => setSuspendDuration(Number(e.target.value))}
              >
                <MenuItem value={7}>7 jours</MenuItem>
                <MenuItem value={14}>14 jours</MenuItem>
                <MenuItem value={30}>30 jours</MenuItem>
              </Select>
            </FormControl>
          </>
        }
        action={
          <Button
            variant="contained"
            color="error"
            onClick={handleSuspendUser}
            disabled={isSuspending || !suspendReason.trim() || !suspendDuration || loading}
            startIcon={isSuspending ? <CircularProgress size={20} /> : null}
          >
            {isSuspending ? 'Suspension...' : 'Suspendre'}
          </Button>
        }
      />

      {/* Dialog de réactivation */}
      <ConfirmDialog
        open={confirmReactivate.value}
        onClose={confirmReactivate.onFalse}
        title={`Réactiver l'utilisateur : ${row.firstName} ${row.lastName}`}
        content="Êtes-vous sûr de vouloir réactiver cet utilisateur suspendu ?"
        action={
          <Button
            variant="contained"
            color="success"
            onClick={handleReactivateUser}
            disabled={isReactivating || loading}
            startIcon={isReactivating ? <CircularProgress size={20} /> : null}
          >
            {isReactivating ? 'Réactivation...' : 'Réactiver'}
          </Button>
        }
      />

      {/* Dialog de confirmation de blocage des enfants */}
      <ConfirmDialog
        open={confirmBlockChildren.value}
        onClose={handleCancelBlockChildren}
        title="Bloquer également les enfants ?"
        content={
          parentInfo
            ? `Souhaitez-vous également bloquer tous les enfants de ${parentInfo.firstName} ${parentInfo.lastName} ?`
            : ''
        }
        action={
          <Button
            variant="contained"
            color="error"
            onClick={handleBlockChildren}
            disabled={isBlockingChildren || loading}
            startIcon={isBlockingChildren ? <CircularProgress size={20} /> : null}
          >
            {isBlockingChildren ? 'Blocage...' : 'Oui, bloquer'}
          </Button>
        }
      />

      {/* Dialog de confirmation de suspension des enfants */}
      <ConfirmDialog
        open={confirmSuspendChildren.value}
        onClose={handleCancelSuspendChildren}
        title="Suspendre également les enfants ?"
        content={
          parentInfo
            ? `Souhaitez-vous également suspendre tous les enfants de ${parentInfo.firstName} ${parentInfo.lastName} ?`
            : ''
        }
        action={
          <Button
            variant="contained"
            color="error"
            onClick={handleSuspendChildren}
            disabled={isSuspendingChildren || loading}
            startIcon={isSuspendingChildren ? <CircularProgress size={20} /> : null}
          >
            {isSuspendingChildren ? 'Suspension...' : 'Oui, suspendre'}
          </Button>
        }
      />

      {/* Dialog de confirmation de réactivation des enfants */}
      <ConfirmDialog
        open={confirmReactivateChildren.value}
        onClose={handleCancelReactivateChildren}
        title="Réactiver les enfants du parent ?"
        content={
          parentInfo
            ? `Le parent ${parentInfo.firstName} ${parentInfo.lastName} a été réactivé. Souhaitez-vous également réactiver tous ses enfants ?`
            : ''
        }
        action={
          <Button
            variant="contained"
            color="success"
            onClick={handleReactivateChildren}
            disabled={isReactivatingChildren || loading}
            startIcon={isReactivatingChildren ? <CircularProgress size={20} /> : null}
          >
            {isReactivatingChildren ? 'Réactivation...' : 'Oui, réactiver'}
          </Button>
        }
      />
    </>
  );
}
