import { useState, useCallback } from 'react';

import Box from '@mui/material/Box';
import Chip from '@mui/material/Chip';
import Stack from '@mui/material/Stack';
import Avatar from '@mui/material/Avatar';
import { alpha } from '@mui/material/styles';
import TextField from '@mui/material/TextField';
import Autocomplete from '@mui/material/Autocomplete';

import Iconify from '@/shared/components/iconify';
import SearchNotFound from '@/shared/components/search-not-found';

import {  IClient } from '@/shared/types/chat';
import { useGetContactsTest } from '@/shared/api/chat javalin';
//

// ----------------------------------------------------------------------

type Props = {
  children:React.ReactNode;
  onAddRecipients: (selected: IClient[]) => void;
};

export default function ChatHeaderCompose({ children, onAddRecipients }: Props) {

  const [searchRecipients, setSearchRecipients] = useState('');
  const { contacts, contactsLoading, contactsError, contactsEmpty } = useGetContactsTest();

  const handleAddRecipients = useCallback(
    (selected: IClient[]) => {
      setSearchRecipients('');
      onAddRecipients(selected);
    },
    [onAddRecipients]
  );
  

  


  return (
    <>
        {children}
      <Autocomplete
        sx={{ minWidth: 240 }}
        multiple
        limitTags={3}
        popupIcon={null}
        defaultValue={[]}
        disableCloseOnSelect
        noOptionsText={<SearchNotFound query={searchRecipients} />}
        onChange={(event, newValue) => handleAddRecipients(newValue)}
        onInputChange={(event, newValue) => setSearchRecipients(newValue)}
        options={contacts}
        getOptionLabel={(recipient) => recipient.firstname}
        isOptionEqualToValue={(option, value) => option.id === value.id}
        renderInput={(params) => <TextField {...params} placeholder="+ Tap to add" />}
        renderOption={(props, recipient, { selected }) => (
          <li {...props} key={recipient.uuid}>
            <Box
              key={recipient.uuid}
              sx={{
                mr: 1,
                width: 32,
                height: 32,
                overflow: 'hidden',
                borderRadius: '50%',
                position: 'relative',
              }}
            >
              <Avatar alt={recipient.firstname} src={recipient.avatarUrl} sx={{ width: 1, height: 1 }} ></Avatar>
              <Stack
                alignItems="center"
                justifyContent="center"
                sx={{
                  top: 0,
                  left: 0,
                  width: 1,
                  height: 1,
                  opacity: 0,
                  position: 'absolute',
                  bgcolor: (theme) => alpha(theme.palette.grey[900], 0.8),
                  transition: (theme) =>
                    theme.transitions.create(['opacity'], {
                      easing: theme.transitions.easing.easeInOut,
                      duration: theme.transitions.duration.shorter,
                    }),
                  ...(selected && {
                    opacity: 1,
                    color: 'primary.main',
                  }),
                }}
              >
                <Iconify icon="eva:checkmark-fill" />
              </Stack>
            </Box>
             
            {recipient.firstname+" "+recipient.lastname}
          </li>
        )}
        renderTags={(selected, getTagProps) =>
          selected.map((recipient, index) => (
            <Chip
              {...getTagProps({ index })}
              key={recipient.uuid}
              label={recipient.firstname}
              avatar={<Avatar alt={recipient.firstname} src={recipient.avatarUrl} />}
              size="small"
              variant="soft"
            />
          ))
        }
      />
    </>
  );
}
