// src/hooks/use-move-document.ts
import { useCallback } from 'react';
import { useSnackbar } from '@/shared/components/snackbar';
import { fileManagerApi } from '@/shared/api/file-manager';
import { getUserByEmail } from '@/shared/api/user';
import { DestinationFolder, SourceDocument } from '@/shared/types/file';



export function useMoveDocument(onSuccess?: VoidFunction) {
  const { enqueueSnackbar } = useSnackbar();

  const moveDocument = useCallback(async (
    sourceDocument: SourceDocument, 
    destinationFolder: DestinationFolder
  ) => {
    try {
      const email = localStorage.getItem('email');
      if (!email) {
        throw new Error('No email found in localStorage');
      }

      const userData = await getUserByEmail(email);
      
      // Utiliser le bon format de chemin
      const sourcePath = sourceDocument.type === 'file' 
        ? sourceDocument.name 
        : sourceDocument.path?.replace(/\\/g, '/') || '';
      
      const destinationPath = destinationFolder.path.replace(/\\/g, '/');
      
      await fileManagerApi.moveDocument(
        Number(userData.id),
        sourcePath,
        destinationPath
      );

      enqueueSnackbar('Document moved successfully');
      
      if (onSuccess) {
        onSuccess();
      }
      
      return true;
    } catch (error) {
      console.error('Error moving document:', error);
      enqueueSnackbar('Error moving document', { variant: 'error' });
      return false;
    }
  }, [enqueueSnackbar, onSuccess]);

  return { moveDocument };
}