import { formatDistanceToNowStrict } from 'date-fns';

import Box from '@mui/material/Box';
import Stack from '@mui/material/Stack';
import Avatar from '@mui/material/Avatar';
import IconButton from '@mui/material/IconButton';
import Typography from '@mui/material/Typography';

import { useMockedUser } from '@/hooks/use-mocked-user';

import Iconify from '@/shared/components/iconify';

import { IChatMessage, IChatParticipant, IClient } from '@/shared/types/chat';

import useGetMessage from '@/hooks/use-get-message';
import { useSnackbar } from '@/components/snackbar';


// ----------------------------------------------------------------------

type Props = {
  message: IChatMessage;
  participants: IClient[];
  onOpenLightbox: (value: string) => void;
};

export default function ChatMessageItem({ message, participants, onOpenLightbox }: Props) {
  const { user1 } = useMockedUser();
  const { enqueueSnackbar } = useSnackbar();
  const { me, senderDetails, hasImage } = useGetMessage({
    message,
    participants,
    currentUserId: `${user1?.id}`,
  });
  let dynamicStyles = {};
  if (me) {
    dynamicStyles = {
      left: 'unset',
      right: 0,
    };
  }

  const { firstName, avatarUrl } = senderDetails;

  const { body, createdAt } = message;
  const handleCopy = () => {
    enqueueSnackbar('Copy success!');
  };

  const handleInject = () => {
    enqueueSnackbar('Inject success!');
  };

  const handleRegenerate = () => {

  };
  const renderInfo = (
    <Typography
      noWrap
      variant="caption"
      sx={{
        mb: 1,
        color: 'text.disabled',
        ...(!me && {
          mr: 'auto',
        }),
      }}
    >
      {!me && `Stunar AI,`} &nbsp;
      {formatDistanceToNowStrict(new Date(createdAt), {
        addSuffix: true,
      })}
    </Typography>
  );

  const renderBody = (
    <Stack
      sx={{
        p: 1.5,
        minWidth: 48,
        maxWidth: 320,
        borderRadius: 1,
        typography: 'body2',
        bgcolor: 'background.neutral',
        ...(me && {
          color: 'grey.800',
          bgcolor: 'primary.lighter',
        }),
        ...(hasImage && {
          p: 0,
          bgcolor: 'transparent',
        }),
      }}
    >
      {hasImage ? (
        <Box
          component="img"
          alt="attachment"
          src={body}
          onClick={() => onOpenLightbox(body)}
          sx={{
            minHeight: 220,
            borderRadius: 1.5,
            cursor: 'pointer',
            '&:hover': {
              opacity: 0.9,
            },
          }}
        />
      ) : (
        body
      )}
    </Stack>
  );

  const renderActions = (
    (!me &&
      <Stack
        direction="row"
        className="message-actions"
        sx={{
          pt: 0.5,
          opacity: 0,
          top: '100%',
          left: 0,
          position: 'absolute',
          transition: (theme) =>
            theme.transitions.create(['opacity'], {
              duration: theme.transitions.duration.shorter,
            }),
          ...dynamicStyles,
        }}
      >

        <IconButton size="small" title="Copy" onClick={handleCopy}>
          <Iconify icon="mingcute:copy-fill" width={16} />
        </IconButton>
        <IconButton size="small" title="Inject the description" onClick={handleInject}>
          <Iconify icon="tabler:replace" width={16} />
        </IconButton>
        <IconButton size="small" title="Regenerate" onClick={handleRegenerate}>
          <Iconify icon="ri:find-replace-line" width={16} />
        </IconButton>


      </Stack>)
  );

  return (
    <Stack direction="row" justifyContent={me ? 'flex-end' : 'unset'} sx={{ mb: 5 }}>
      {!me && <Avatar alt='' src={'/logo/ST-AI.png'} sx={{ width: 30, height: 30, mr: 2 }} />}

      <Stack alignItems="flex-end">
        {renderInfo}

        <Stack
          direction="row"
          alignItems="center"
          sx={{
            position: 'relative',
            '&:hover': {
              '& .message-actions': {
                opacity: 1,
              },
            },
          }}
        >
          {renderBody}
          {renderActions}
        </Stack>
      </Stack>
    </Stack>
  );
}
