import { useState } from 'react';
import {
  Dialog,
  DialogTitle,
  DialogContent,
  DialogActions,
  Stack,
  Typography,
  List,
  ListItem,
  ListItemIcon,
  ListItemText,
  IconButton,
  Button,
  Tooltip,
  CircularProgress,
} from '@mui/material';
import { useRouter } from '@/hooks';
import { useSnackbar } from '@/components/snackbar';
import Iconify from '@/shared/components/iconify';
import { paths } from '@/routes/paths';

interface UrlsDialogProps {
  open: boolean;
  urls: Record<string, string>;
  onClose: () => void;
  isLoading?: boolean;
}

export default function UrlsDialog({ 
  open, 
  urls, 
  onClose, 
  isLoading = false 
}: UrlsDialogProps) {
  const [copiedUrl, setCopiedUrl] = useState<string | null>(null);
  const { enqueueSnackbar } = useSnackbar();
  const router = useRouter();

  const handleCopyUrl = async (url: string) => {
    try {
      await navigator.clipboard.writeText(url);
      setCopiedUrl(url);
      enqueueSnackbar('Link copied!');
      setTimeout(() => setCopiedUrl(null), 2000);
    } catch (error) {
      enqueueSnackbar('Failed to copy link', { variant: 'error' });
    }
  };

  const handleClose = () => {
    onClose();
    router.push(paths.dashboard.transfer.root);
  };

  return (
    <Dialog open={open} onClose={onClose} maxWidth="md" fullWidth>
      <DialogTitle>
        <Stack direction="row" alignItems="center" spacing={1}>
          <Iconify icon="eva:link-2-fill" />
          <Typography variant="h6">
            {isLoading ? 'Loading URLs...' : 'Download Links'}
          </Typography>
        </Stack>
      </DialogTitle>

      <DialogContent>
        {isLoading ? (
          <Stack alignItems="center" justifyContent="center" sx={{ py: 3 }}>
            <CircularProgress />
          </Stack>
        ) : (
          <List>
            {Object.entries(urls).map(([fileName, url]) => (
              <ListItem
                key={fileName}
                sx={{
                  borderRadius: 1,
                  mb: 1,
                  bgcolor: 'background.neutral',
                }}
              >
                <ListItemIcon>
                  <Iconify icon="eva:file-fill" />
                </ListItemIcon>

                <ListItemText
                  primary={fileName}
                  secondary={
                    <Typography
                      variant="body2"
                      sx={{
                        overflow: 'hidden',
                        textOverflow: 'ellipsis',
                        color: 'text.secondary',
                      }}
                    >
                      {url}
                    </Typography>
                  }
                />

                <Tooltip title="Copy link">
                  <IconButton
                    onClick={() => handleCopyUrl(url)}
                    color={copiedUrl === url ? 'primary' : 'default'}
                  >
                    <Iconify 
                      icon={copiedUrl === url ? 'eva:checkmark-fill' : 'eva:copy-fill'} 
                    />
                  </IconButton>
                </Tooltip>
              </ListItem>
            ))}
          </List>
        )}
      </DialogContent>

      <DialogActions>
        <Button 
          variant="outlined" 
          onClick={handleClose}
          disabled={isLoading}
        >
          Done
        </Button>
      </DialogActions>
    </Dialog>
  );
}