import { forwardRef, useMemo } from 'react';
import { usePathname } from 'next/navigation';

import Box from '@mui/material/Box';
import Link from '@mui/material/Link';
import Tooltip from '@mui/material/Tooltip';
import { styled, alpha } from '@mui/material/styles';
import ListItemButton from '@mui/material/ListItemButton';

import RouterLink from '@/routes/router-link';
import Iconify from '../../iconify';
import { NavItemProps, NavItemStateProps } from '../types';

// ----------------------------------------------------------------------

const NavItem = forwardRef<HTMLDivElement, NavItemProps>(
  (
    {
      title,
      path,
      icon,
      info,
      disabled,
      caption,
      roles,
      //
      open,
      depth,
      active,
      hasChild,
      externalLink,
      currentRole = 'admin',
      ...other
    },
    ref
  ) => {
    const subItem = depth !== 1;
    const pathname = usePathname();

    const isActive = useMemo(() => {
      if (path === '/dashboard') {
        return pathname === '/dashboard';
      }
      if (hasChild) {
        return pathname.startsWith(path);
      }
      return pathname === path;
    }, [path, pathname, hasChild]);

    const renderContent = (
      <StyledNavItem
        ref={ref}
        open={open}
        depth={depth}
        active={isActive}
        disabled={disabled}
        data-active={isActive}
        {...other}
      >
        {icon && (
          <Box component="span" className="icon">
            {icon}
          </Box>
        )}

        {title && (
          <Box component="span" className="label">
            {title}
          </Box>
        )}

        {caption && (
          <Tooltip title={caption} arrow>
            <Iconify width={16} icon="eva:info-outline" className="caption" />
          </Tooltip>
        )}

        {info && (
          <Box component="span" className="info">
            {info}
          </Box>
        )}

        {hasChild && (
          <Iconify
            width={16}
            className="arrow"
            icon={subItem ? 'eva:arrow-ios-forward-fill' : 'eva:arrow-ios-downward-fill'}
          />
        )}
      </StyledNavItem>
    );

    // Hidden item by role
    if (roles && !roles.includes(`${currentRole}`)) {
      return null;
    }

    if (externalLink)
      return (
        <Link
          href={path}
          target="_blank"
          rel="noopener"
          color="inherit"
          underline="none"
          sx={{
            ...(disabled && {
              cursor: 'default',
            }),
          }}
        >
          {renderContent}
        </Link>
      );

    return (
      <Link
        component={RouterLink}
        href={path}
        color="inherit"
        underline="none"
        sx={{
          ...(disabled && {
            cursor: 'default',
          }),
        }}
      >
        {renderContent}
      </Link>
    );
  }
);

export default NavItem;

// ----------------------------------------------------------------------

const StyledNavItem = styled(ListItemButton, {
  shouldForwardProp: (prop) => prop !== 'active' && prop !== 'open',
})<NavItemStateProps>(({ active, open, depth, theme }) => {
  const subItem = depth !== 1;
  const opened = open && !active;

  const baseStyles = {
    item: {
      ...theme.typography.body2,
      borderRadius: 6,
      color: theme.palette.text.secondary,
      fontWeight: theme.typography.fontWeightMedium,
      transition: theme.transitions.create(['background-color', 'border-color', 'color'], {
        duration: theme.transitions.duration.shorter,
      }),
    },
    icon: {
      width: 22,
      height: 22,
      flexShrink: 0,
      marginRight: theme.spacing(1),
    },
    label: {
      textTransform: 'capitalize',
    },
    caption: {
      marginLeft: theme.spacing(0.75),
      color: theme.palette.text.disabled,
    },
    info: {
      display: 'inline-flex',
      marginLeft: theme.spacing(0.75),
    },
    arrow: {
      marginLeft: theme.spacing(0.75),
    },
  } as const;

  return {
    // Root item
    ...(!subItem && {
      ...baseStyles.item,
      minHeight: 32,
      flexShrink: 0,
      padding: theme.spacing(0, 0.75),
      borderRadius: 1,
      border: '1px solid transparent',
      '& .icon': {
        ...baseStyles.icon,
      },
      '& .label': {
        ...baseStyles.label,
      },
      '& .caption': {
        ...baseStyles.caption,
      },
      '& .info': {
        ...baseStyles.info,
      },
      '& .arrow': {
        ...baseStyles.arrow,
      },
      '&[data-active="true"]': {
        color: '#144dd5',
        backgroundColor: alpha('#144dd5', 0.08),
        border: `1px solid #144dd5`,
        fontWeight: theme.typography.fontWeightSemiBold,
        '&:hover': {
          backgroundColor: alpha('#144dd5', 0.16),
          borderColor: alpha('#144dd5', 0.32),
        },
        '& .icon': {
          color: '#144dd5',
        },
      },
      '&:hover': {
        backgroundColor: alpha('#144dd5', 0.08),
        color: '#144dd5',
        borderColor: alpha('#144dd5', 0.24),
        '& .icon': {
          color: '#144dd5',
        },
      },
    }),

    // Sub item
    ...(subItem && {
      ...baseStyles.item,
      minHeight: 34,
      padding: theme.spacing(0, 1),
      '& .icon': {
        ...baseStyles.icon,
      },
      '& .label': {
        ...baseStyles.label,
        flexGrow: 1,
      },
      '& .caption': {
        ...baseStyles.caption,
      },
      '& .info': {
        ...baseStyles.info,
      },
      '& .arrow': {
        ...baseStyles.arrow,
        marginRight: theme.spacing(-0.5),
      },
      '&[data-active="true"]': {
        color: '#144dd5',
        backgroundColor: alpha('#144dd5', 0.08),
        borderColor: alpha('#144dd5', 0.24),
        fontWeight: theme.typography.fontWeightSemiBold,
        '&:hover': {
          backgroundColor: alpha('#144dd5', 0.16),
          borderColor: alpha('#144dd5', 0.32),
        },
        '& .icon': {
          color: '#144dd5',
        },
      },
      '&:hover': {
        backgroundColor: alpha('#144dd5', 0.08),
        color: '#144dd5',
        borderColor: alpha('#144dd5', 0.24),
        '& .icon': {
          color: '#144dd5',
        },
      },
    }),
  };
});