import { Dialog, IconButton, Button, Box } from '@mui/material';
import CloseIcon from '@mui/icons-material/Close';
import DownloadIcon from '@mui/icons-material/Download';
import ConditionalComponent from '../ConditionalComponent';

// ----------------------------------------------------------------------

type Props = {
  type: string;
  open: boolean;
  onClose: VoidFunction;
  url: string | undefined;
}

const DOC_EXTENSIONS = ['pdf', 'doc', 'docx', 'txt', 'rtf', 'odt', 'xlsx', 'xls', 'csv', 'ppt', 'pptx', 'DOCUMENT', 'PDF'];

export default function DocPlayer({ type, url, open, onClose }: Props) {
  if (!url) return null;
  
  const isDocType = DOC_EXTENSIONS.includes(type) ||
    DOC_EXTENSIONS.includes(type.toLowerCase()) ||
    type.toLowerCase() === 'document' ||
    type.toLowerCase() === 'pdf';

  const getViewerUrl = () => {
    if (type.toLowerCase() === 'pdf' || (url && url.toLowerCase().endsWith('.pdf'))) {
      return url;
    }
    return `https://view.officeapps.live.com/op/view.aspx?src=${encodeURIComponent(url)}`;
  };

  const documentViewer = (
    <Dialog
      fullWidth
      maxWidth='lg'
      open={open}
      onClose={onClose}
      PaperProps={{
        sx: {
          backgroundColor: 'transparent',
          height: '90vh',
          position: 'relative',
          overflow: 'hidden'
        }
      }}
    >
      <div style={{
        position: 'relative',
        width: '100%',
        height: '100%',
        backgroundColor: '#f5f5f5',
        paddingTop: '40px',
      }}>
        <IconButton
          onClick={onClose}
          sx={{
            position: 'absolute',
            top: 0,
            right: 8,
            color: 'common.white',
            backgroundColor: 'rgba(0, 0, 0, 0.5)',
            '&:hover': {
              backgroundColor: 'rgba(0, 0, 0, 0.7)',
            },
            zIndex: 9999,
            boxShadow: '0 2px 4px rgba(0,0,0,0.2)'
          }}
        >
          <CloseIcon />
        </IconButton>

        <Box
          sx={{
            position: 'absolute',
            top: 0,
            left: 8,
            zIndex: 9999,
          }}
        >
          <Button
            href={url}
            target="_blank"
            rel="noopener"
            startIcon={<DownloadIcon />}
            sx={{
              color: 'common.white',
              backgroundColor: 'rgba(0, 0, 0, 0.5)',
              '&:hover': {
                backgroundColor: 'rgba(0, 0, 0, 0.7)',
              },
              boxShadow: '0 2px 4px rgba(0,0,0,0.2)'
            }}
          >
            download
          </Button>
        </Box>

        <iframe
          src={getViewerUrl()}
          width='100%'
          height='100%'
          style={{
            border: 'none',
            backgroundColor: 'white',
            borderRadius: '4px'
          }}
          title="Document Viewer"
          allow="fullscreen"
        />
      </div>
    </Dialog>
  );

  return (
    <ConditionalComponent isValid={isDocType}>
      {documentViewer}
    </ConditionalComponent>
  );
}