import { useState, useEffect, useCallback } from 'react';

import Box from '@mui/material/Box';
import Stack from '@mui/material/Stack';
import Drawer from '@mui/material/Drawer';
import { useTheme } from '@mui/material/styles';
import TextField from '@mui/material/TextField';
import IconButton from '@mui/material/IconButton';
import InputAdornment from '@mui/material/InputAdornment';
import ClickAwayListener from '@mui/material/ClickAwayListener';

import { paths } from '@/routes/paths';
import { useRouter } from '@/hooks';

import { useResponsive } from '@/hooks/use-responsive';

import Iconify from '@/shared/components/iconify';

import { IChatParticipant, IChatConversations, IClient } from '@/shared/types/chat';

import { useCollapseNav } from '@/hooks';

import {  Container, Tooltip, Typography } from '@mui/material';
import ScrollDialog from './chat-scroll-dialog';
import ComponentBlock from '../agency/component-block';

// ----------------------------------------------------------------------

const NAV_WIDTH = 500;

const NAV_COLLAPSE_WIDTH = 96;

type Props = {
  loading: boolean;
  selectedConversationId: string;
  contacts: IClient[];
  conversations: IChatConversations;
};

export default function ChatNavGlobal({
  loading,
  contacts,
  conversations,
  selectedConversationId,
}: Props) {

  const router = useRouter();

  const mdUp = useResponsive('up', 'md');

  // État pour stocker l'état des notifications pour chaque conversation
  const [notificationsAct, setNotificationsAct] = useState(true);

  const {
    collapseDesktop,
    onCloseDesktop,
    onCollapseDesktop,
    //
    openMobile,
    onOpenMobile,
    onCloseMobile,
  } = useCollapseNav();

  const [searchContacts, setSearchContacts] = useState<{
    query: string;
    results: IClient[];
  }>({
    query: '',
    results: [],
  });

  useEffect(() => {
    if (!mdUp) {
      onCloseDesktop();
    }
  }, [onCloseDesktop, mdUp]);

  const handleToggleNav = useCallback(() => {
    if (mdUp) {
      onCollapseDesktop();
    } else {
      onCloseMobile();
    }
  }, [mdUp, onCloseMobile, onCollapseDesktop]);

  const handleClickCompose = useCallback(() => {
    if (!mdUp) {
      onCloseMobile();
    }
    router.push(paths.dashboard.chat);
  }, [mdUp, onCloseMobile, router]);

  const handleSearchContacts = useCallback(
    (inputValue: string) => {
      setSearchContacts((prevState) => ({
        ...prevState,
        query: inputValue,
      }));

      if (inputValue) {
        const results = contacts.filter((contact) =>
          contact.firstname.toLowerCase().includes(inputValue)
        );

        setSearchContacts((prevState) => ({
          ...prevState,
          results,
        }));
      }
    },
    [contacts]
  );

  const handleClickAwaySearch = useCallback(() => {
    setSearchContacts({
      query: '',
      results: [],
    });
  }, []);

  const handleClickResult = useCallback(
    (result: IChatParticipant) => {
      handleClickAwaySearch();

      router.push(`${paths.dashboard.chat}?id=${result.id}`);
    },
    [handleClickAwaySearch, router]
  );
  
  const handleNotificationClick = ()=>{
    setNotificationsAct(!notificationsAct);
  }
  const renderSearchInput = (
    <ClickAwayListener onClickAway={handleClickAwaySearch} >
      <TextField
        fullWidth
        variant="standard"
        value={searchContacts.query}
        onChange={(event) => handleSearchContacts(event.target.value)}
        placeholder="Search..."
        InputProps={{
          startAdornment: (
            <InputAdornment position="start">
              <Iconify icon="eva:search-fill" sx={{ color: 'text.disabled' }} />
            </InputAdornment>
          ),
        }}
        sx={{ mt: 2.5 ,height:'50px'}}
      />
    </ClickAwayListener>
  );
  const [openDialog, setOpenDialog] = useState<(() => void) | null>(null);

  return (
    <>
          <Container >
          <Stack
              direction="row" alignItems="center" justifyContent="space-between"
              sx={{marginBottom:1}}
            > 
              <Box>
                <Typography
                  variant="h4"
                  sx={{marginRight:10,bottom:0}}
                >
                  Chat
                </Typography>
                </Box>
              <Stack direction='row' spacing={4} justifyContent='space-between' alignItems='center'>
                <Box >{renderSearchInput}</Box>
                <Stack direction='row' spacing={1} justifyContent='space-between' alignItems='center'>
                {/* <Box sx={{height:'15px',cursor:'pointer'}} ><Iconify icon='fluent:settings-chat-24-filled' sx={{width:'25px',height:'25px'}}/></Box> */}
                <Tooltip title={notificationsAct?'Mutate all':'Activate all'}><Box sx={{height:'15px',cursor:'pointer'}} onClick={handleNotificationClick}><Iconify icon={notificationsAct?"mdi:notifications":"mdi:notifications-off" } sx={{width:'25px',height:'25px'}} /></Box></Tooltip>
                </Stack>
              </Stack>
              
          </Stack>
        </Container>
      
    </>
  );
}
