import { useSettingsContext } from "@/shared/components/settings";
import { 
  Box, 
  Container, 
  Grid, 
  Typography, 
  Card, 
  Stack, 
  Alert, 
  AlertTitle,
  Divider,
  Avatar,
  LinearProgress,
  styled
} from "@mui/material";
import { MotivationIllustration } from "@/shared/assets/illustrations";
import { useTheme } from "@mui/material/styles";
import { CheckCircleOutline, AccessTime, Person } from "@mui/icons-material";

type User = {
  name?: string;
  email?: string;
};

type UserHomeProps = {
  user: User | null;
};

const StyledCard = styled(Card)(({ theme }) => ({
  padding: theme.spacing(4),
  borderRadius: theme.spacing(2),
  boxShadow: theme.shadows[3],
  height: '100%',
  transition: 'transform 0.3s ease-in-out, box-shadow 0.3s ease-in-out',
  '&:hover': {
    transform: 'translateY(-5px)',
    boxShadow: theme.shadows[10],
  },
}));

const WelcomeCard = styled(Card)(({ theme }) => ({
  padding: theme.spacing(5),
  borderRadius: theme.spacing(2),
  boxShadow: theme.shadows[3],
  backgroundImage: `linear-gradient(135deg, ${theme.palette.primary.light}, ${theme.palette.primary.main})`,
  color: theme.palette.primary.contrastText,
  position: 'relative',
  overflow: 'hidden',
  '&::after': {
    content: '""',
    position: 'absolute',
    top: 0,
    right: 0,
    width: '40%',
    height: '100%',
    backgroundImage: `radial-gradient(circle, ${theme.palette.primary.lighter} 20%, transparent 70%)`,
    opacity: 0.2,
  },
}));

export default function UserHome({ user }: UserHomeProps) {
  const settings = useSettingsContext();
  const theme = useTheme();

  if (!user) {
    return (
      <Container maxWidth={settings.themeStretch ? false : "xl"}>
        <Box display="flex" justifyContent="center" alignItems="center" minHeight="60vh">
          <LinearProgress sx={{ width: '50%' }} />
        </Box>
      </Container>
    );
  }

  return (
    <Container maxWidth={settings.themeStretch ? false : "xl"}>
      <Typography
        variant="h4"
        sx={{
          mb: { xs: 3, md: 5 },
          fontWeight: 700,
        }}
      >
        Dashboard
      </Typography>

      <Grid container spacing={3}>
        <Grid item xs={12}>
          <WelcomeCard>
            <Stack direction={{ xs: 'column', md: 'row' }} spacing={3} alignItems="center">
              <Box flex={1}>
                <Typography variant="h4" sx={{ mb: 2, fontWeight: 700 }}>
                  Welcome back, {user.name || "Guest"}! 👋
                </Typography>
                <Typography variant="body1" sx={{ mb: 3, opacity: 0.9 }}>
                  Your account has been successfully authenticated. Your profile is currently under review by our administrators.
                </Typography>
                <Alert 
                  severity="info" 
                  icon={<AccessTime color="info" />}
                  sx={{ 
                    bgcolor: 'rgba(255, 255, 255, 0.15)', 
                    color: 'inherit',
                    border: '1px solid rgba(255, 255, 255, 0.3)'
                  }}
                >
                  <AlertTitle sx={{ fontWeight: 600 }}>Pending Approval</AlertTitle>
                  Your profile is awaiting admin verification. You'll receive full access once approved.
                </Alert>
              </Box>
              <Box 
                sx={{ 
                  display: { xs: 'none', md: 'block' },
                  position: 'relative',
                  zIndex: 1
                }}
              >
                <MotivationIllustration sx={{ height: 200 }} />
              </Box>
            </Stack>
          </WelcomeCard>
        </Grid>
      </Grid>

      <Typography
        variant="h5"
        sx={{
          mt: 6,
          mb: 3,
          fontWeight: 600,
        }}
      >
        What to expect after approval
      </Typography>

      <Grid container spacing={3}>
        <Grid item xs={12} md={6}>
          <StyledCard>
            <Stack spacing={2}>
              <Box sx={{ display: 'flex', alignItems: 'center', mb: 1 }}>
                <CheckCircleOutline color="primary" sx={{ mr: 1 }} />
                <Typography variant="h6" fontWeight={600}>Member Communication</Typography>
              </Box>
              <Typography variant="body2" color="text.secondary">
                Connect and collaborate with team members through our integrated communication platform. Share ideas, updates, and feedback in real-time.
              </Typography>
              <Divider sx={{ my: 1 }} />
              <Typography variant="body2" color="text.secondary">
                Features include direct messaging, group chats, and notification systems to keep everyone in sync.
              </Typography>
            </Stack>
          </StyledCard>
        </Grid>
        
        <Grid item xs={12} md={6}>
          <StyledCard>
            <Stack spacing={2}>
              <Box sx={{ display: 'flex', alignItems: 'center', mb: 1 }}>
                <CheckCircleOutline color="primary" sx={{ mr: 1 }} />
                <Typography variant="h6" fontWeight={600}>Project Management</Typography>
              </Box>
              <Typography variant="body2" color="text.secondary">
                Organize and track projects with our comprehensive management tools. Set milestones, assign tasks, and monitor progress all in one place.
              </Typography>
              <Divider sx={{ my: 1 }} />
              <Typography variant="body2" color="text.secondary">
                Includes Gantt charts, kanban boards, and deadline tracking to ensure project success.
              </Typography>
            </Stack>
          </StyledCard>
        </Grid>
        
        <Grid item xs={12} md={6}>
          <StyledCard>
            <Stack spacing={2}>
              <Box sx={{ display: 'flex', alignItems: 'center', mb: 1 }}>
                <CheckCircleOutline color="primary" sx={{ mr: 1 }} />
                <Typography variant="h6" fontWeight={600}>Invoice Management</Typography>
              </Box>
              <Typography variant="body2" color="text.secondary">
                Create, send, and track invoices effortlessly. Our system automates payment reminders and provides clear financial reporting.
              </Typography>
              <Divider sx={{ my: 1 }} />
              <Typography variant="body2" color="text.secondary">
                Supports multiple currencies, tax calculations, and integration with popular payment gateways.
              </Typography>
            </Stack>
          </StyledCard>
        </Grid>
        
        <Grid item xs={12} md={6}>
          <StyledCard>
            <Stack spacing={2}>
              <Box sx={{ display: 'flex', alignItems: 'center', mb: 1 }}>
                <CheckCircleOutline color="primary" sx={{ mr: 1 }} />
                <Typography variant="h6" fontWeight={600}>File Storage</Typography>
              </Box>
              <Typography variant="body2" color="text.secondary">
                Access a secure, shared file repository for your team. Store, organize, and collaborate on documents with version control.
              </Typography>
              <Divider sx={{ my: 1 }} />
              <Typography variant="body2" color="text.secondary">
                Includes permission settings, file locking, and advanced search capabilities to find what you need quickly.
              </Typography>
            </Stack>
          </StyledCard>
        </Grid>
      </Grid>

      <Box mt={6} mb={3} textAlign="center">
        <Stack direction="row" spacing={2} justifyContent="center" alignItems="center">
          <Avatar sx={{ bgcolor: theme.palette.primary.main }}>
            <Person />
          </Avatar>
          <Typography variant="body2" color="text.secondary">
            Logged in as <strong>{user.email}</strong>
          </Typography>
        </Stack>
      </Box>
    </Container>
  );
}