import React from 'react';
import {
  Stack,
  Button,
  Tooltip,
  IconButton,
  Dialog,
  DialogActions,
  Box,
} from '@mui/material';
import { PDFDownloadLink, PDFViewer } from '@react-pdf/renderer';
import CircularProgress from '@mui/material/CircularProgress';
import Iconify from '@/shared/components/iconify';
import { ConfirmDialog } from '@/shared/components/custom-dialog';
import InvoicePDF from '@/shared/sections/invoice/invoice-pdf';
import { IInvoice } from '@/shared/types/invoice';

interface InvoiceListActionsProps {
  selected: string[];
  onDeleteRows: () => void;
  tableData: IInvoice[];
  view: {
    value: boolean;
    onTrue: () => void;
    onFalse: () => void;
  };
  confirm: {
    value: boolean;
    onTrue: () => void;
    onFalse: () => void;
  };
}

export default function InvoiceListActions({
  selected,
  onDeleteRows,
  tableData,
  view,
  confirm,
}: InvoiceListActionsProps) {
  const selectedInvoice = selected.length === 1 ? tableData.find((invoice) => invoice.id === selected[0]) : null;
  return (
    <>
      <Stack direction="row" spacing={2}>
        {selectedInvoice ? (
          <>
            <PDFDownloadLink
              document={
                <InvoicePDF
                  invoice={selectedInvoice}
                  currentStatus={selectedInvoice.status || ''}
                />
              }
              fileName={selectedInvoice.invoiceNumber || 'invoice.pdf'}
              style={{ textDecoration: 'none' }}
            >
              {({ loading }) => (
                <Tooltip title="Download">
                  <IconButton color="primary">
                    {loading ? (
                      <CircularProgress size={24} color="inherit" />
                    ) : (
                      <Iconify icon="eva:cloud-download-fill" />
                    )}
                  </IconButton>
                </Tooltip>
              )}
            </PDFDownloadLink>

            <Tooltip title="Print">
              <IconButton color="primary" onClick={view.onTrue}>
                <Iconify icon="solar:printer-minimalistic-bold" />
              </IconButton>
            </Tooltip>

            <Tooltip title="Delete">
              <IconButton color="primary" onClick={confirm.onTrue}>
                <Iconify icon="solar:trash-bin-trash-bold" />
              </IconButton>
            </Tooltip>
          </>
        ) : (
          <Tooltip title="Select an invoice to see actions">
            <span>
              <IconButton color="primary" disabled>
                <Iconify icon="eva:info-outline" />
              </IconButton>
            </span>
          </Tooltip>
        )}
      </Stack>

      <Dialog fullScreen open={view.value}>
        <Box sx={{ height: 1, display: 'flex', flexDirection: 'column' }}>
          <DialogActions
            sx={{
              p: 1.5,
            }}
          >
            <Button color="inherit" variant="contained" onClick={view.onFalse}>
              Close
            </Button>
          </DialogActions>

          <Box sx={{ flexGrow: 1, height: 1, overflow: 'hidden' }}>
            <PDFViewer width="100%" height="100%" style={{ border: 'none' }}>
              {selected.length === 1 ? (
                tableData.find((invoice) => invoice.id === selected[0]) ? (
                  <InvoicePDF
                    invoice={tableData.find((invoice) => invoice.id === selected[0])!}
                    currentStatus={
                      tableData.find((invoice) => invoice.id === selected[0])?.status || ''
                    }
                  />
                ) : (
                  <></>
                )
              ) : (
                <></>
              )}
            </PDFViewer>
          </Box>
        </Box>
      </Dialog>

      <ConfirmDialog
        open={confirm.value}
        onClose={confirm.onFalse}
        title="Delete"
        content={
          <>
            Are you sure want to delete <strong> {selected.length} </strong> items?
          </>
        }
        action={
          <Button variant="contained" color="error" onClick={onDeleteRows}>
            Delete
          </Button>
        }
      />
    </>
  );
}