import List from '@mui/material/List';
import Button from '@mui/material/Button';
import TextField from '@mui/material/TextField';
import DialogTitle from '@mui/material/DialogTitle';
import DialogActions from '@mui/material/DialogActions';
import DialogContent from '@mui/material/DialogContent';
import InputAdornment from '@mui/material/InputAdornment';
import Dialog, { DialogProps } from '@mui/material/Dialog';
import Select from '@mui/material/Select';
import MenuItem from '@mui/material/MenuItem';
import FormControl from '@mui/material/FormControl';
import InputLabel from '@mui/material/InputLabel';
import Stack from '@mui/material/Stack';

import Iconify from '@/shared/components/iconify';
import Scrollbar from '@/shared/components/scrollbar';
import { useSnackbar } from '@/shared/components/snackbar';

import { IFileShared } from '@/shared/types/file';
import { AccessLevel } from '@/shared/api/file-manager';

import FileManagerInvitedItem from './file-manager-invited-item';
import { useState } from 'react';
import { Typography } from '@mui/material';

type Props = DialogProps & {
  inviteEmail?: string;
  shared?: IFileShared[] | null;
  onCopyLink?: VoidFunction;
  onChangeInvite?: (event: React.ChangeEvent<HTMLInputElement>) => void;
  onShare?: (email: string, accessLevel: AccessLevel) => Promise<void>;
  uniqueId?: string;
  open: boolean;
  onClose: VoidFunction;
};

export default function FileManagerShareDialog({
  shared,
  inviteEmail,
  onCopyLink,
  onChangeInvite,
  onShare,
  uniqueId,
  open,
  onClose,
  ...other
}: Props) {
  const { enqueueSnackbar } = useSnackbar();
  const [accessLevel, setAccessLevel] = useState<AccessLevel>(AccessLevel.READ);
  const hasShared = shared && !!shared.length;

  const handleShare = async () => {
    if (!inviteEmail || !onShare) return;

    try {
      await onShare(inviteEmail, accessLevel);
      enqueueSnackbar('Document shared successfully');
      onClose();
    } catch (error) {
      console.error('Error sharing document:', error);
      enqueueSnackbar('Error sharing document', { variant: 'error' });
    }
  };

  const renderInviteField = onChangeInvite && (
    <TextField
      fullWidth
      value={inviteEmail}
      label="Email"
      onChange={onChangeInvite}
      placeholder="recipient@example.com"
    />
  );

  const renderAccessLevelSelect = (
    <FormControl fullWidth>
      <InputLabel>Access Level</InputLabel>
      <Select
        value={accessLevel}
        label="Access Level"
        onChange={(e) => setAccessLevel(e.target.value as AccessLevel)}
      >
        
        <MenuItem value={AccessLevel.READ}>Read</MenuItem>
        <MenuItem value={AccessLevel.WRITE}>Write</MenuItem>
      </Select>
    </FormControl>
  );

  const renderSharedList = hasShared && (
    <>
      <Typography variant="subtitle1" sx={{ mt: 4, mb: 2 }}>
        Shared With
      </Typography>
      <Scrollbar sx={{ maxHeight: 60 * 6 }}>
        <List disablePadding>
          {shared.map((person) => (
            <FileManagerInvitedItem key={person.id} person={person} />
          ))}
        </List>
      </Scrollbar>
    </>
  );

  const renderCopyLinkButton = onCopyLink && (
    <Button 
      startIcon={<Iconify icon="eva:link-2-fill" />} 
      onClick={onCopyLink}
      variant="outlined"
    >
      Copy link
    </Button>
  );

  return (
    <Dialog fullWidth maxWidth="xs" open={open} onClose={onClose} {...other}>
      <DialogTitle>Share Document</DialogTitle>

      <DialogContent sx={{ overflow: 'unset' }}>
        <Stack spacing={2} sx={{ mt: 2 }}>
          {renderInviteField || (
            <Typography color="text.secondary" align="center">
              Email input not available
            </Typography>
          )}
          
          {renderAccessLevelSelect || (
            <Typography color="text.secondary" align="center">
              Access level selection not available
            </Typography>
          )}

          <Button
            fullWidth
            variant="contained"
            disabled={!inviteEmail}
            onClick={handleShare}
          >
            Share Document
          </Button>
        </Stack>

        {renderSharedList}
      </DialogContent>

      <DialogActions sx={{ justifyContent: 'space-between', px: 3, pb: 3 }}>
        {renderCopyLinkButton || (
          <Typography variant="body2" color="text.secondary">
            Sharing link not available
          </Typography>
        )}

        <Button variant="outlined" color="inherit" onClick={onClose}>
          Close
        </Button>
      </DialogActions>
    </Dialog>
  );
}