import Box from "@mui/material/Box";
import Card from "@mui/material/Card";
import Stack from "@mui/material/Stack";
import Divider from "@mui/material/Divider";
import MenuItem from "@mui/material/MenuItem";
import IconButton from "@mui/material/IconButton";
import Typography from "@mui/material/Typography";
import ListItemText from "@mui/material/ListItemText";
import Iconify from "@/shared/components/iconify";
import CustomPopover, { usePopover } from "@/shared/components/custom-popover";
import { fDate } from "@/utils/format-time";
import { IRewardItem } from "@/contexts/types/reward";

// ----------------------------------------------------------------------

type Props = {
  reward: IRewardItem; // Reward data object
  onView: VoidFunction;
  onEdit: VoidFunction;
  onDelete: VoidFunction;
};

export default function RewardItem({
  reward,
  onView,
  onEdit,
  onDelete,
}: Props) {
  const popover = usePopover();

  // Destructure reward data for simplicity
  const {
    id,
    title,
    description,
    startDate,
    endDate,
    rule,
    status,
    authorName,
  } = reward;

  return (
    <>
      <Card sx={{ position: "relative", boxShadow: 3, borderRadius: 2 }}>
        {/* More Options Button */}
        <IconButton
          onClick={popover.onOpen}
          sx={{ position: "absolute", top: 8, right: 8 }}
        >
          <Iconify icon="eva:more-vertical-fill" />
        </IconButton>

        {/* Title */}
        <ListItemText
          sx={{ mb: 1, mt: 2, ml: 2 }}
          primary={
            <Typography variant="subtitle1">{title || "No Title"}</Typography>
          }
          secondary={
            <Typography variant="caption" color="text.secondary">
              {description || "No description available"}
            </Typography>
          }
        />

        <Divider sx={{ borderStyle: "dashed" }} />

        {/* Reward Details */}
        <Stack sx={{ p: 3, pb: 2 }}>
          <Box
            display="grid"
            gridTemplateColumns="repeat(2, 1fr)"
            columnGap={2}
          >
            {/* Start and End Dates */}
            <Box>
              <Stack direction="row" alignItems="center" spacing={1}>
                <Iconify width={18} icon="tabler:calendar-due" />
                <Typography variant="caption">
                  {startDate && !isNaN(new Date(startDate).getTime())
                    ? fDate(startDate)
                    : "Invalid Start Date"}
                </Typography>
              </Stack>
              <Stack
                direction="row"
                alignItems="center"
                spacing={1}
                sx={{ color: "error.main" }}
              >
                <Iconify width={18} icon="solar:bell-bing-bold-duotone" />
                <Typography variant="caption">
                  {endDate && !isNaN(new Date(endDate).getTime())
                    ? fDate(endDate)
                    : "Invalid End Date"}
                </Typography>
              </Stack>
            </Box>

            {/* Trophy Icon */}
            <Box display="flex" justifyContent="center" alignItems="center">
              <Iconify
                icon="solar:cup-star-bold"
                sx={{
                  width: 40,
                  height: 40,
                  color: "primary.main",
                  bgcolor: "primary.lighter",
                  borderRadius: "50%",
                  p: 1,
                }}
              />
            </Box>
          </Box>
        </Stack>

        <Divider sx={{ borderStyle: "dashed" }} />

        {/* Additional Details */}
        <Box sx={{ p: 3 }}>
          <Stack direction="row" spacing={2}>
            {/* Author */}
            <Stack direction="row" alignItems="center" spacing={1}>
              <Iconify width={18} icon="solar:users-group-rounded-bold" />
              <Typography variant="caption">{authorName || 'No Author'}</Typography>

            </Stack>

            {/* Rule */}
            <Stack direction="row" alignItems="center" spacing={1}>
              <Iconify
                width={18}
                icon="material-symbols:rule-settings-rounded"
              />
              <Typography variant="caption">{rule || "No Rule"}</Typography>
            </Stack>
          </Stack>
        </Box>
      </Card>

      {/* Popover Menu */}
      <CustomPopover
        open={popover.open}
        onClose={popover.onClose}
        arrow="right-top"
        sx={{ width: 140 }}
      >
        <MenuItem
          onClick={() => {
            popover.onClose();
            onView();
          }}
        >
          <Iconify icon="solar:eye-bold" />
          View
        </MenuItem>
        <MenuItem
          onClick={() => {
            popover.onClose();
            onEdit();
          }}
        >
          <Iconify icon="mdi:pencil" />
          Edit
        </MenuItem>
        <MenuItem
          onClick={() => {
            popover.onClose();
            onDelete();
          }}
          sx={{ color: "error.main" }}
        >
          <Iconify icon="solar:trash-bin-trash-bold" />
          Delete
        </MenuItem>
      </CustomPopover>
    </>
  );
}
