import Avatar from "@mui/material/Avatar";
import MenuItem from "@mui/material/MenuItem";
import TableRow from "@mui/material/TableRow";
import Checkbox from "@mui/material/Checkbox";
import TableCell from "@mui/material/TableCell";
import IconButton from "@mui/material/IconButton";
import ListItemText from "@mui/material/ListItemText";
import Divider from "@mui/material/Divider";
import Button from "@mui/material/Button";

import { useBoolean } from "@/hooks/use-boolean";
import Label from "@/components/label";
import Iconify from "@/shared/components/iconify";
import { ConfirmDialog } from "@/components/custom-dialog";
import CustomPopover, { usePopover } from "@/shared/components/custom-popover";
import ConfirmDialogDeleteUser from "../confirm-dialog-delete-user";
import { IUserItem } from "@/shared/types/user";

type Props = {
  selected: boolean;
  admin?: boolean;
  onEditRow: VoidFunction;
  row: IUserItem;
  onSelectRow: VoidFunction;
  onDeleteRow: VoidFunction;
  onBlock: VoidFunction;
};

export default function UserTableRow({
  row,
  selected,
  admin,
  onEditRow,
  onSelectRow,
  onDeleteRow,
  onBlock,
}: Props) {
  const { firstName, lastName, email, phoneNumber, role, status, agency } = row;
  const confirm = useBoolean();
  const confirm2 = useBoolean();
  const popover = usePopover();

  const isBlocked = status === "blocked"; // Check if the user is currently blocked

  return (
    <>
      <TableRow hover selected={selected}>
        {/* Checkbox */}
        <TableCell padding="checkbox">
          <Checkbox checked={selected} onClick={onSelectRow} />
        </TableCell>

        {/* Name Column */}
        <TableCell sx={{ display: "flex", alignItems: "center" }}>
          <Avatar
            alt={`${firstName} ${lastName}`}
            src="/static/default-avatar.png"
            sx={{ mr: 2 }}
          />
          <ListItemText
            primary={`${firstName} ${lastName}`}
            primaryTypographyProps={{ typography: "body2" }}
          />
        </TableCell>

        {/* Email Column */}
        <TableCell sx={{ whiteSpace: "nowrap" }}>{email || "N/A"}</TableCell>

        {/* Phone Number Column */}
        <TableCell sx={{ whiteSpace: "nowrap" }}>
          {phoneNumber || "N/A"}
        </TableCell>

        {/* Role Column */}
        <TableCell sx={{ whiteSpace: "nowrap" }}>{role || "N/A"}</TableCell>

        {/* Agency Name Column */}
        <TableCell sx={{ whiteSpace: "nowrap" }}>{agency || "N/A"}</TableCell>

        {/* Status Column */}
        <TableCell>
          <Label
            variant="soft"
            color={
              (status === "active" && "success") ||
              (status === "blocked" && "error") ||
              "default"
            }
          >
            {status}
          </Label>
        </TableCell>

        {/* Action Buttons */}
        <TableCell align="right" sx={{ px: 1, whiteSpace: "nowrap" }}>
          <IconButton
            color={popover.open ? "inherit" : "default"}
            onClick={popover.onOpen}
          >
            <Iconify icon="eva:more-vertical-fill" />
          </IconButton>
        </TableCell>
      </TableRow>

      {/* Popover for actions like Block/Unblock, Edit, Delete */}
      <CustomPopover
        open={popover.open}
        onClose={popover.onClose}
        arrow="right-top"
        sx={{ width: 140 }}
      >
        {/* Block/Unblock Button */}
        <MenuItem
          onClick={() => {
            onBlock();
            popover.onClose();
          }}
          sx={{ color: isBlocked ? "success.main" : "error.main" }} // Green for unblock, red for block
        >
          <Iconify
            icon={isBlocked ? "fa-solid:unlock" : "fa-solid:lock"} // Icon changes
            sx={{ color: isBlocked ? "success.main" : "error.main" }}
          />
          {isBlocked ? "Unblock" : "Block"} {/* Label changes */}
        </MenuItem>

        <Divider sx={{ borderStyle: "dashed" }} />

        {/* Delete User Option */}
        <MenuItem
          onClick={() => {
            confirm2.onTrue();
            popover.onClose();
          }}
          sx={{ color: "error.main" }}
        >
          <Iconify icon="solar:trash-bin-trash-bold" />
          Delete
        </MenuItem>
      </CustomPopover>

      {/* Confirm Dialog for Blocking */}
      <ConfirmDialog
        open={confirm.value}
        onClose={confirm.onFalse}
        title={isBlocked ? "Unblock" : "Block"}
        content={`Are you sure you want to ${isBlocked ? "unblock" : "block"}?`}
        action={
          <Button
            variant="contained"
            color={isBlocked ? "success" : "error"}
            onClick={() => {
              onBlock();
              confirm.onFalse();
            }}
          >
            {isBlocked ? "Unblock" : "Block"}
          </Button>
        }
      />

      {/* Confirm Dialog for Deleting */}
      <ConfirmDialogDeleteUser
        open={confirm2.value}
        onClose={confirm2.onFalse}
        title="Delete"
        content="Are you sure you want to delete?"
        action={
          <Button
            variant="contained"
            color="error"
            onClick={() => {
              onDeleteRow();
              confirm2.onFalse();
            }}
          >
            Delete
          </Button>
        }
      />
    </>
  );
}
