import Box from '@mui/material/Box';
import Table from '@mui/material/Table';
import Tooltip from '@mui/material/Tooltip';
import { useTheme } from '@mui/material/styles';
import TableBody from '@mui/material/TableBody';
import IconButton from '@mui/material/IconButton';
import TableContainer from '@mui/material/TableContainer';
import { tableCellClasses } from '@mui/material/TableCell';
import { tablePaginationClasses } from '@mui/material/TablePagination';

import Iconify from '@/shared/components/iconify';
import { TableProps, TableSelectedAction } from '@/components/table/index';
import { TableNoData } from '@/components/table/index';
import { TableHeadCustom } from '@/components/table/index';
import { TablePaginationCustom } from '@/components/table/index';
import { IProjectItem, Project } from '@/contexts/types/Project';



import { ConfirmDialog } from '@/components/custom-dialog';
import { Button } from '@mui/material';
import { useBoolean } from '@/hooks';
import ProjectArchivedTableRow from './project-archived-table-row';

// ----------------------------------------------------------------------

const TABLE_HEAD = [
    { id: 'name', label: 'Name' },
    { id: 'type', label: 'Type', width: 120 },
    { id: 'Satuts', label: 'Status', width: 120 },
    { id: 'EndDate', label: 'End Date', width: 140 },
    { id: 'StartDate', label: 'Start Date', width: 140 },
    { id: 'Participants', label: 'Participants', align: 'right', width: 140 },
    { id: '', width: 88 },


];

// ----------------------------------------------------------------------

type Props = {
    table: TableProps;
    notFound: boolean;
    dataFiltered: Project[];
    onOpenConfirm1: VoidFunction;
    onOpenConfirm: VoidFunction;
    onDeleteRow: (id: number) => void;
    onRecoverRow: (id: number) => void;
};

export default function ProjectArchivedTable({
    table,
    notFound,
    onDeleteRow,
    dataFiltered,
    onRecoverRow,
    onOpenConfirm1,
    onOpenConfirm,
}: Props) {
    const theme = useTheme();

    const {
        dense,
        page,
        order,
        orderBy,
        rowsPerPage,
        //
        selected,
        onSelectRow,
        onSelectAllRows,
        //
        onSort,
        onChangeDense,
        onChangePage,
        onChangeRowsPerPage,
    } = table;
    const confirmRecover = useBoolean();
    return (
        <>
            <Box
                sx={{
                    position: 'relative',
                    m: theme.spacing(-2, -3, -3, -3),
                }}
            >
                <TableSelectedAction
                    dense={dense}
                    numSelected={selected.length}
                    rowCount={dataFiltered.length}
                    onSelectAllRows={(checked: any) =>
                        onSelectAllRows(
                            checked,
                            dataFiltered.map((row) => row.id.toString())
                        )
                    }
                    action={
                        <>
                            <Tooltip title="Share">
                                <IconButton color="primary">
                                    <Iconify icon="ic:baseline-share" />
                                </IconButton>
                            </Tooltip>

                            <Tooltip title="Delete">
                                <IconButton color="primary" onClick={onOpenConfirm}>
                                    <Iconify icon="solar:trash-bin-trash-bold" />
                                </IconButton>
                            </Tooltip>
                            <Tooltip title="Recover">
                                <IconButton color="primary" onClick={onOpenConfirm1}>
                                    <Iconify icon="humbleicons:arrow-go-back" />
                                </IconButton>
                            </Tooltip>
                        </>
                    }
                    sx={{
                        pl: 1,
                        pr: 2,
                        top: 16,
                        left: 24,
                        right: 24,
                        width: 'auto',
                        borderRadius: 1.5,
                    }}
                />


                <TableContainer
                    sx={{
                        p: theme.spacing(0, 3, 3, 3),
                    }}
                >
                    <Table
                        size={dense ? 'small' : 'medium'}
                        sx={{
                            minWidth: 960,
                            borderCollapse: 'separate',
                            borderSpacing: '0 16px',
                        }}
                    >
                        <TableHeadCustom
                            order={order}
                            orderBy={orderBy}
                            headLabel={TABLE_HEAD}
                            rowCount={dataFiltered.length}
                            numSelected={selected.length}
                            onSort={onSort}
                            onSelectAllRows={(checked) =>
                                onSelectAllRows(
                                    checked,
                                    dataFiltered.map((row) => row.id.toString())
                                )
                            }
                            sx={{
                                [`& .${tableCellClasses.head}`]: {
                                    '&:first-of-type': {
                                        borderTopLeftRadius: 12,
                                        borderBottomLeftRadius: 12,
                                    },
                                    '&:last-of-type': {
                                        borderTopRightRadius: 12,
                                        borderBottomRightRadius: 12,
                                    },
                                },
                            }}
                        />

                        <TableBody>
                            {dataFiltered
                                .slice(page * rowsPerPage, page * rowsPerPage + rowsPerPage)
                                .map((row) => (
                                    <><ProjectArchivedTableRow
                                        key={row.id}
                                        row={row}
                                        selected={selected.includes(row.id.toString())}
                                        onSelectRow={() => onSelectRow(row.id.toString())}
                                        onDeleteRow={() => onDeleteRow(row.id)}
                                        onRecoverRow={() => onRecoverRow(row.id)} />
                                    </>
                                ))}

                            <TableNoData
                                notFound={notFound}
                                sx={{
                                    m: -2,
                                    borderRadius: 1.5,
                                    border: `dashed 1px ${theme.palette.divider}`,
                                }}
                            />
                        </TableBody>
                    </Table>
                    <ConfirmDialog
                        open={confirmRecover.value}
                        onClose={confirmRecover.onFalse}
                        title="Recover"
                        content="Are you sure want to recover this task?"
                        action={
                            <Button variant="contained" color="success" onClick={confirmRecover.onFalse}>
                                Recover
                            </Button>
                        }
                    />
                </TableContainer>
            </Box>


            <TablePaginationCustom
                count={dataFiltered.length}
                page={page}
                rowsPerPage={rowsPerPage}
                onPageChange={onChangePage}
                onRowsPerPageChange={onChangeRowsPerPage}
                dense={dense}
                onChangeDense={onChangeDense}
                sx={{
                    [`& .${tablePaginationClasses.toolbar}`]: {
                        borderTopColor: 'transparent',
                    },
                }}
            />
        </>
    );
}
