import Stack from '@mui/material/Stack';
import Badge from '@mui/material/Badge';
import Avatar from '@mui/material/Avatar';
import IconButton from '@mui/material/IconButton';
import ListItemText from '@mui/material/ListItemText';
import AvatarGroup, { avatarGroupClasses } from '@mui/material/AvatarGroup';

import { fToNow } from '@/utils/format-time';
import {useGetNavItem, useMockedUser, useResponsive, useSearchParams } from '@/hooks';
import { IChatConversation, IChatConversationClient, IChatParticipant, IClient } from '@/shared/types/chat';
import { useTheme } from '@mui/material/styles';
import { usePopover } from '@/shared/components/custom-popover';
import { Typography } from '@mui/material';

// ----------------------------------------------------------------------

type Props = {
  participants: IClient[];
  selectedId:string;
  children:React.ReactNode;
  conversation: IChatConversationClient;

};

export default function ChatHeaderDetail({ participants,children,conversation }: Props) {
  
  const { user1 } = useMockedUser();
  
  // const group = participants?.length > 2;
  const singleParticipant = participants?.filter(participant => participant.uuid !== user1.uuid)[0];
  const multipleParticipants =  participants?.filter(participant => participant.uuid !== user1.uuid);

  const mdUp = useResponsive('up', 'md');
  

  const {displayName, displayText,lastActivity, hasOnlineInGroup} =
  useGetNavItem({
    conversation,
    currentUserId: `${user1?.id}`,
  });

  const popover = usePopover();  

  const isChannel = conversation?.typeConv === 'CHANNEL';

  const renderGroup = (
    <>
    {mdUp && <AvatarGroup
      max={3}
      sx={{
        [`& .${avatarGroupClasses.avatar}`]: {
          width: 32,
          height: 32,
        },
      }}
    >
      {participants?.map((participant) => (
        <Avatar key={participant.id} alt={participant.firstname} src={participant.avatarUrl} />
      ))}
     
    </AvatarGroup>}
    <Stack direction="column" alignItems="flex-start" sx={{width:1}} spacing={0} >
      <Typography variant="subtitle1">
          {conversation?.name
            ? conversation.name
            : participants.map((participant) => participant.firstname).join(', ')}
        </Typography>
      {mdUp && <ListItemText
            sx={{width:'300px'}}
            primary={displayName}
            primaryTypographyProps={{
              noWrap: true,
              variant: conversation?.unreadCount ? 'subtitle2' : 'body2',
              color: conversation?.unreadCount ? 'text.primary' : 'text.secondary',
            }}
          />}
    </Stack>
    </>
  );

  const renderSingle = (
    <Stack flexGrow={1} direction="row" alignItems="center" spacing={2}>
      <Badge
        variant={singleParticipant?.status}
        anchorOrigin={{ vertical: 'bottom', horizontal: 'right' }}
      >
        <Avatar src={singleParticipant?.avatarUrl} alt={singleParticipant?.lastname} >{singleParticipant?.firstname[0].toUpperCase()}</Avatar>
      </Badge>

      <ListItemText
        primary={singleParticipant?.firstname+" "+singleParticipant?.lastname}
        secondary={
          singleParticipant?.status !== 'online'
            ? fToNow(singleParticipant?.lastActivity)
            : singleParticipant?.status
        }
        secondaryTypographyProps={{
          component: 'span',
          ...(singleParticipant?.status === 'online' && {
            textTransform: 'capitalize',
          }),
        }}
      />
    </Stack>
  );
  

  



  return (
    <>
    <Stack direction='row' alignContent={'flex-start'} alignItems={'center'} spacing={1} >
    {isChannel ? renderGroup : renderSingle}
    </Stack>

      <Stack flexGrow={1} />
       
       {children}
      
      
    </>
  );
}
