import { useState, useCallback } from 'react';

import Box from '@mui/material/Box';
import Chip from '@mui/material/Chip';
import Stack from '@mui/material/Stack';
import Avatar from '@mui/material/Avatar';
import { alpha } from '@mui/material/styles';
import TextField from '@mui/material/TextField';
import Autocomplete from '@mui/material/Autocomplete';

import Iconify from '@/shared/components/iconify';
import SearchNotFound from '@/shared/components/search-not-found';

import { IChatMessage, IChatParticipant } from '@/shared/types/chat';
import { Button, Dialog, DialogActions, DialogContent, DialogProps, DialogTitle } from '@mui/material';
//

// ----------------------------------------------------------------------

type Props = {
  contacts: IChatParticipant[];
  onAddRecipients: (selected: IChatParticipant[]) => void;
  message: IChatMessage | null;
  
};

interface PropsDialog extends DialogProps {
    title?: string;
    //
    onCreate?: VoidFunction;
    onUpdate?: VoidFunction;
    //
    folderName?: string;
    //
    open: boolean;
    onClose: VoidFunction;
  }
  type CombinedProps = Props & PropsDialog;
export default function ChatMessageTransfert({ contacts,onAddRecipients,message ,onClose,open,...other}:CombinedProps) {

  const [searchRecipients, setSearchRecipients] = useState('');
  const [recipients,setRecipients] = useState<IChatParticipant[]>([])

  const handleAddRecipients = useCallback(
    (selected: IChatParticipant[]) => {
      setSearchRecipients('');
      onAddRecipients(selected);
      setRecipients(selected);
    },
    [onAddRecipients]
  );
  return (
    <>
    <Dialog fullWidth maxWidth="sm" open={open} onClose={onClose} {...other}>
        <DialogTitle sx={{ p: (theme) => theme.spacing(3, 3, 2, 3) }}> Share message with : </DialogTitle>

        <DialogContent dividers sx={{ pt: 1, pb: 1, border: 'none' }}>
            <Autocomplete
            sx={{ minWidth: 240 }}
            multiple
            limitTags={3}
            popupIcon={null}
            defaultValue={[]}
            disableCloseOnSelect
            noOptionsText={<SearchNotFound query={searchRecipients} />}
            onChange={(event, newValue) => handleAddRecipients(newValue)}
            onInputChange={(event, newValue) => setSearchRecipients(newValue)}
            options={contacts}
            getOptionLabel={(recipient) => recipient.name}
            isOptionEqualToValue={(option, value) => option.id === value.id}
            renderInput={(params) => <TextField {...params} placeholder="+ Select recipients" />}
            renderOption={(props, recipient, { selected }) => (
            <li {...props} key={recipient.id}>
                <Box
                key={recipient.id}
                sx={{
                    mr: 1,
                    width: 32,
                    height: 32,
                    overflow: 'hidden',
                    borderRadius: '50%',
                    position: 'relative',
                }}
                >
                <Avatar alt={recipient.name} src={recipient.avatarUrl} sx={{ width: 1, height: 1 }} />
                <Stack
                    alignItems="center"
                    justifyContent="center"
                    sx={{
                    top: 0,
                    left: 0,
                    width: 1,
                    height: 1,
                    opacity: 0,
                    position: 'absolute',
                    bgcolor: (theme) => alpha(theme.palette.grey[900], 0.8),
                    transition: (theme) =>
                        theme.transitions.create(['opacity'], {
                        easing: theme.transitions.easing.easeInOut,
                        duration: theme.transitions.duration.shorter,
                        }),
                    ...(selected && {
                        opacity: 1,
                        color: 'primary.main',
                    }),
                    }}
                >
                    <Iconify icon="eva:checkmark-fill" />
                </Stack>
                </Box>
                
                {recipient.name}
            </li>
            )}
            renderTags={(selected, getTagProps) =>
            selected.map((recipient, index) => (
                <Chip
                {...getTagProps({ index })}
                key={recipient.id}
                label={recipient.name}
                avatar={<Avatar alt={recipient.name} src={recipient.avatarUrl} />}
                size="small"
                variant="soft"
                />
            ))
            }
        />
        </DialogContent>
        <DialogActions>
        
            <Button
            variant="contained"
            startIcon={<Iconify icon="carbon:send-alt" width={18} />}
            disabled={!recipients.length}
            // onClick={handleUpload}
            >
                Send
            </Button>
            
            
            <Button variant="outlined" onClick={onClose}>
                Cancel
            </Button>
        </DialogActions>
    </Dialog>
    </>
  );
}
