'use client';

import { Box, styled, Paper, ListItem, ListItemIcon, ListItemText } from '@mui/material';
import { alpha, useTheme } from '@mui/material/styles';
import { paths } from '@/routes/paths';
import Iconify from '@/shared/components/iconify';
import { useRouter, usePathname } from 'next/navigation';
import { useEffect, useState } from 'react';

const NavSection = styled(Paper)(({ theme }) => ({
  width: 80,
  minHeight: '100vh',
  position: 'fixed',
  left: 0,
  top: 'var(--banner-height, 0px)',
  backgroundColor: theme.palette.background.paper,
  borderRight: `1px solid ${alpha(theme.palette.divider, 0.05)}`,
  paddingTop: theme.spacing(8),
  display: 'flex',
  flexDirection: 'column',
  transition: 'top 0.3s ease',
}));

const ContentSection = styled(Box)(({ theme }) => ({
  marginLeft: 80,
  width: 'calc(100% - 80px)',
}));

const NAV_ITEMS = [
  {
    title: 'Invoices',
    path: paths.dashboard.invoice.root,
    icon: 'solar:bill-list-outline',
    matchPaths: [
      paths.dashboard.invoice.root,
      paths.dashboard.invoice.details,
      paths.dashboard.invoice.edit,
      paths.dashboard.invoice.grid,
      paths.dashboard.invoice.new,
    ]
  },
  {
    title: 'Reports',
    path: paths.dashboard.invoice.analytics,
    icon: 'solar:chart-outline',
    matchPaths: [
      paths.dashboard.invoice.analytics
    ]
  },
  {
    title: 'Subscription',
    path: paths.dashboard.invoice.new,
    icon: 'solar:diploma-verified-outline',
    matchPaths: [
      paths.dashboard.invoice.new
    ]
  },
];

type InvoiceLayoutProps = {
  children: React.ReactNode;
};

export default function InvoiceLayout({ children }: InvoiceLayoutProps) {
  const theme = useTheme();
  const router = useRouter();
  const pathname = usePathname();
  const [showBanner] = useState(true);
  const isPathSelected = (item: typeof NAV_ITEMS[0]) => {
    if (pathname.includes('/analytics')) {
      return item.path === paths.dashboard.invoice.analytics;
    }
    if (pathname.includes('/new')) {
      return item.path === paths.dashboard.invoice.new;
    }
    if (item.path === paths.dashboard.invoice.root) {
      return pathname === paths.dashboard.invoice.root || 
             (pathname.includes('/invoice') && 
              !pathname.includes('/analytics') && 
              !pathname.includes('/new'));
    }
    
    return false;
  };
  useEffect(() => {
    document.documentElement.style.setProperty(
      '--banner-height',
      showBanner ? '40px' : '0px'
    );
  }, [showBanner]);

  const handleNavigation = (path: string) => {
    router.push(path);
  };

  return (
    <Box sx={{ display: 'flex' }}>
      <NavSection elevation={0} sx={{
         
          top: showBanner ? 40 : 0,
        }}>
        <Box 
          sx={{ 
            display: 'flex',
            flexDirection: 'column',
            mt: 4, 
            gap: 4,
          }}
        >
          {NAV_ITEMS.map((item) => (
            <ListItem
              key={item.path}
              button
              onClick={() => handleNavigation(item.path)}
              sx={{
                flexDirection: 'column',
                alignItems: 'center',
                justifyContent: 'center',
                height: 90,
                width: '100%',
                position: 'relative',
                cursor: 'pointer',
                '&:before': {
                  content: '""',
                  position: 'absolute',
                  left: 0,
                  top: 0,
                  bottom: 0,
                  width: 3,
                  bgcolor: '#4F80EE',
                  opacity: isPathSelected(item) ? 1 : 0,
                  transition: theme.transitions.create('opacity'),
                },
                '&:hover': {
                  bgcolor: 'transparent',
                  '& .MuiTypography-root': { 
                    color: '#4F80EE',
                  },
                  '& .MuiListItemIcon-root': { 
                    color: '#4F80EE',
                  },
                },
                ...(isPathSelected(item) && {
                  '& .MuiTypography-root': { 
                    color: '#4F80EE',
                    fontWeight: 600,
                  },
                  '& .MuiListItemIcon-root': { 
                    color: '#4F80EE',
                  },
                }),
              }}
            >
              <ListItemIcon
                sx={{
                  minWidth: 'auto',
                  color: isPathSelected(item) ? '#4F80EE' : 'text.disabled',
                  justifyContent: 'center',
                  mb: 1.5,
                  transition: theme.transitions.create(['color']),
                }}
              >
                <Iconify icon={item.icon} width={24} />
              </ListItemIcon>
              <ListItemText
                primary={item.title}
                primaryTypographyProps={{
                  noWrap: true,
                  fontSize: 13,
                  textAlign: 'center',
                  color: isPathSelected(item) ? '#4F80EE' : 'text.disabled',
                  fontWeight: isPathSelected(item) ? 600 : 400,
                  sx: { 
                    transition: theme.transitions.create(['color', 'font-weight']),
                  }
                }}
              />
            </ListItem>
          ))}
        </Box>
      </NavSection>
      <ContentSection>
        {children}
      </ContentSection>
    </Box>
  );
}