import { useState } from 'react';
import Stack from '@mui/material/Stack';
import Button from '@mui/material/Button';
import { useTheme } from '@mui/material/styles';
import InputBase from '@mui/material/InputBase';
import Box, { BoxProps } from '@mui/material/Box';
import { bgGradient } from '@/shared/theme/css';
import axios from 'axios';
import { useSnackbar } from 'notistack';
import axiosInstancee from '@/utils/axios';

// ----------------------------------------------------------------------

interface Props extends BoxProps {
  title?: string;
  description?: string;
  rewardId: number;
  onSuccess?: (updatedTask: any) => void;
}

export default function AddMarkToTasks({
  title,
  description,
  rewardId,
  onSuccess,
  sx,
  ...other
}: Props) {
  const theme = useTheme();
  const [mark, setMark] = useState<number | ''>('');
  const [isSubmitting, setIsSubmitting] = useState(false);
  const { enqueueSnackbar } = useSnackbar();

  const handleAddMark = () => {
    if (!mark || mark < 0) {
      enqueueSnackbar('Please enter a valid mark.', { variant: 'error' });
      return;
    }

    setIsSubmitting(true);

    axiosInstancee.post(`/api/projects/${rewardId}/add-mark`, { mark }, {
      headers: {
        'Content-Type': 'application/json',
      }
    })
    .then((response) => {
      enqueueSnackbar('Mark added successfully!', { variant: 'success' });
      setMark('');
      if (onSuccess) {
        onSuccess(response.data);
      }
    })
    .catch((error) => {
      console.error('Error adding mark:', error);
      enqueueSnackbar(error.response?.data?.message || 'Failed to add mark', { 
        variant: 'error' 
      });
    })
    .finally(() => {
      setIsSubmitting(false);
    });
  };

  return (
    <Box {...other}>
      <Box
        sx={{
          mt: 2,
          color: 'common.white',
          borderRadius: 2,
          p: theme.spacing(4, 3),
          ...bgGradient({
            direction: '135deg',
            startColor: theme.palette.primary.main,
            endColor: theme.palette.primary.dark,
          }),
        }}
      >
        <Stack direction="row" alignItems="center" justifyContent="space-between">
          <Box sx={{ whiteSpace: 'pre-line', typography: 'h6' }}>{title}</Box>
        </Stack>

        <Box sx={{ mt: 2, mb: 3, typography: 'body2' }}>{description}</Box>

        <InputBase
          fullWidth
          type="number"
          value={mark}
          onChange={(e) => {
            const value = Number(e.target.value);
            setMark(value);
          }}
          inputProps={{ 
            min: 0, 
            step: "1"
          }}
          placeholder="Enter mark"
          endAdornment={
            <Button
              color="warning"
              variant="contained"
              size="small"
              sx={{ mr: 0.5 }}
              onClick={handleAddMark}
              disabled={isSubmitting}
            >
              {isSubmitting ? 'Adding...' : 'Add'}
            </Button>
          }
          sx={{
            pl: 1.5,
            height: 40,
            borderRadius: 1,
            bgcolor: 'common.white',
          }}
        />
      </Box>
    </Box>
  );
}