import { IRewardItem } from "@/contexts/types/reward";
import Iconify from "@/shared/components/iconify";
import { fDate } from "@/utils/format-time";
import {
  Box,
  Button,
  Card,
  Divider,
  ListItemText,
  Stack,
  Typography,
} from "@mui/material";
import Markdown from "react-markdown";

type Props = {
  reward: IRewardItem;
  selected: boolean;
  onSelected: VoidFunction;
};

export default function RewardDetailsContent({ reward, selected, onSelected }: Props) {

  const handleToggleStatus = () => {
    const newStatus = reward.status === "PUBLISHED" ? "DRAFT" : "PUBLISHED";
    console.log(`Toggling status to: ${newStatus}`);
  };
  
  const renderHead = (
    <Stack direction="row" sx={{ mb: -3 }}>
      <Typography variant="h4" sx={{ flexGrow: 1 }}>
        {reward.title}
      </Typography>
      <Button
  size="small"
  variant={reward.status === "PUBLISHED" ? "text" : "outlined"}
  color={reward.status === "PUBLISHED" ? "info" : "inherit"}
  startIcon={<Iconify width={18} icon="solar:cup-star-bold" sx={{ mr: -0.75 }} />}
  onClick={handleToggleStatus}
>
  {reward.status === "PUBLISHED" ? "Published" : "Draft"}
</Button>
    </Stack>
  );

  const renderOverview = (
    <Box
      gap={3}
      display="grid"
      gridTemplateColumns={{
        xs: "repeat(1, 1fr)",
        md: "repeat(2, 1fr)",
      }}
    >
      {[
        {
          label: "Available",
          value: `${fDate(reward.startDate)} - ${fDate(reward.endDate)}`,
          icon: <Iconify icon="solar:calendar-date-bold" />,
        },
        {
          label: "Author Name",
          value: reward.authorName,
          icon: <Iconify icon="solar:user-rounded-bold" />,
        },
        {
          label: "Durations",
          value: reward.startDate && reward.endDate 
            ? `${Math.ceil((new Date(reward.endDate).getTime() - new Date(reward.startDate).getTime()) / (1000 * 60 * 60 * 24))} days` 
            : "N/A",
          icon: <Iconify icon="solar:clock-circle-bold" />,
        },
        {
          label: "Author Phone",
          value: reward.authorPhoneNumber,
          icon: <Iconify icon="solar:phone-bold" />,
        },
        {
          label: "Author Email",
          value: reward.authorEmail,
          icon: <Iconify icon="ic:round-email" />,
        },
        {
          label: "Rule",
          value: reward.rule,
          icon: <Iconify icon="material-symbols:rule-settings" />,
        },
      ].map((item) => (
        <Stack key={item.label} spacing={1.5} direction="row">
          {item.icon}
          <ListItemText
            primary={item.label}
            secondary={item.value}
            primaryTypographyProps={{
              typography: "body2",
              color: "text.secondary",
              mb: 0.5,
            }}
            secondaryTypographyProps={{
              typography: "subtitle2",
              color: "text.primary",
              component: "span",
            }}
          />
        </Stack>
      ))}
    </Box>
  );

  return (
    <Stack sx={{ mx: "auto" }}>
      <Card sx={{ p: 4 }}>
        {renderHead}
        <Divider sx={{ borderStyle: "dashed", my: 5 }} />
        {renderOverview}
        <Divider sx={{ borderStyle: "dashed", my: 5 }} />
        <Typography variant="h4" sx={{ flexGrow: 1, mt: -3 }}>
          Description
        </Typography>
        <Markdown>{reward.description}</Markdown>
      </Card>
    </Stack>
  );
}