import Chart, { useChart } from '@/shared/components/chart';
import Box from '@mui/material/Box';
import Typography from '@mui/material/Typography';

interface Props {
  series: {
    name: string;
    data: number[];
    axis: string[];
    color: string;
  }[];
  searchValue: string;
  height?: number;
}

export default function ChartColumnMultiple({ series, searchValue = '', height = 320 }: Props) {
  // Normalize the searchValue to avoid runtime errors
  const normalizedSearchValue = searchValue.trim().toLowerCase();

  // Filter the series based on the search value
  const filteredSeries = series
    .map((item) => {
      const filteredAxis = item.axis.filter((axisItem) =>
        axisItem.toLowerCase().includes(normalizedSearchValue)
      );
      const filteredData = filteredAxis.map((axisItem) => {
        const index = item.axis.indexOf(axisItem);
        return item.data[index];
      });
      return {
        ...item,
        axis: filteredAxis,
        data: filteredData,
      };
    })
    .filter((item) => item.axis.length > 0);

  // Determine the series to use (all data or filtered data)
  const displaySeries = normalizedSearchValue === '' ? series : filteredSeries;

  // If no data to display, render a message
  if (displaySeries.length === 0 || displaySeries[0].data.length === 0) {
    return (
      <Box sx={{ 
        p: 3, 
        textAlign: 'center', 
        height: height, 
        display: 'flex', 
        alignItems: 'center', 
        justifyContent: 'center',
        width: '100%'
      }}>
        <Typography variant="body2" color="text.secondary">
          No data to display
        </Typography>
      </Box>
    );
  }

  // Chart options (use for all cases)
  const chartOptions = useChart({
    chart: {
      zoom: {
        enabled: true, // Enable zoom functionality
        type: 'x', // Zoom along the x-axis
        autoScaleYaxis: true, // Automatically adjust Y-axis scale
      },
      height: height,
    },
    plotOptions: {
      bar: {
        columnWidth: '60%', // Adjust column width for better spacing
      },
    },
    xaxis: {
      categories: displaySeries[0].axis, // Use categories from the series
      labels: {
        rotate: -45, // Rotate labels for readability
        style: {
          fontSize: '10px', // Adjust font size for clarity
        },
      },
    },
    yaxis: {
      labels: {
        style: {
          fontSize: '10px', // Adjust font size for Y-axis
        },
      },
    },
    tooltip: {
      y: {
        formatter: (value: number) => `${value}`, // Tooltip format
      },
    },
    dataLabels: {
      enabled: false, // Disable data labels for a cleaner look
    },
  });

  // Transform series for chart component
  const chartSeries = displaySeries.map(item => ({
    name: item.name,
    data: item.data,
    color: item.color
  }));

  // Render the chart
  return (
    <Box sx={{ width: '100%', height: height }}>
      <Chart
        dir="ltr"
        type="bar"
        series={chartSeries}
        options={chartOptions}
        width="100%"
        height={height}
      />
    </Box>
  );
}