import { Box, ListItemButton, ListItemText, Stack, Typography } from '@mui/material';
import { formatDistanceToNow } from 'date-fns';
import Iconify from '@/shared/components/iconify';

interface NotificationItemProps {
  notification: {
    dateTime: string;
    title: string;
    message: string;
    type: string | null;
    isRead: boolean;
  };
  onMarkAsRead: () => void;
}

export default function NotificationItem({ notification, onMarkAsRead }: NotificationItemProps) {
  const { title, message, dateTime, isRead } = notification;

  return (
    <ListItemButton
      onClick={onMarkAsRead}
      sx={{
        py: 1.5,
        px: 2.5,
        mt: '1px',
        ...(isRead && {
          bgcolor: 'action.selected',
        }),
      }}
    >
      <ListItemText
        primary={
          <Stack direction="row" alignItems="center" justifyContent="space-between">
            <Typography variant="subtitle2">
              {title}
            </Typography>
            {!isRead && (
              <Box
                sx={{
                  width: 8,
                  height: 8,
                  borderRadius: '50%',
                  bgcolor: 'info.main',
                }}
              />
            )}
          </Stack>
        }
        secondary={
          <Stack direction="column" sx={{ mt: 0.5 }}>
            <Typography variant="body2" color="text.secondary">
              {message}
            </Typography>
            <Typography variant="caption" sx={{ color: 'text.disabled', mt: 0.5 }}>
              <Iconify icon="eva:clock-fill" width={16} sx={{ mr: 0.5, verticalAlign: 'sub' }} />
              {formatDistanceToNow(new Date(dateTime), { addSuffix: true })}
            </Typography>
          </Stack>
        }
      />
    </ListItemButton>
  );
}