import { ConfirmDialog } from "@/components/custom-dialog";
import { useBoolean } from "@/hooks";
import CustomPopover, { usePopover } from "@/shared/components/custom-popover";
import Iconify from "@/shared/components/iconify";
import { IAgency } from "@/shared/types/agency";
import { fDate } from "@/utils/format-time";
import { Avatar, AvatarGroup, Button, Checkbox, Divider, IconButton, ListItemText, MenuItem, TableCell, TableRow, Typography, avatarGroupClasses } from "@mui/material";

type Props = {
  row: IAgency;
  selected: boolean;
  onSelectRow: VoidFunction;
  onViewRow: VoidFunction;
  onEditRow: VoidFunction;
  onDeleteRow: VoidFunction;
};

export default function InvoiceTableRow({
  row,
  selected,
  onSelectRow,
  onViewRow,
  onEditRow,
  onDeleteRow,
}: Props) {
  const { name, createDate, users, phoneNumber } = row;

  const confirm = useBoolean();
  const popover = usePopover();

  return (
    <>
      <TableRow hover selected={selected}>
        <TableCell padding="checkbox">
          <Checkbox checked={selected} onClick={onSelectRow} />
        </TableCell>

        <TableCell sx={{ display: 'flex', alignItems: 'center' }}>
          <Avatar alt={name} sx={{ mr: 2 }}>
            {name.charAt(0).toUpperCase()}
          </Avatar>
          <ListItemText
            disableTypography
            primary={<Typography variant="body2" noWrap>{name}</Typography>}
          />
        </TableCell>

        <TableCell>
          <ListItemText
            primary={fDate(createDate)}
            primaryTypographyProps={{ typography: 'body2', noWrap: true }}
            secondaryTypographyProps={{
              mt: 0.5,
              component: 'span',
              typography: 'caption',
            }}
          />
        </TableCell>

        {/* Display Phone Number */}
        <TableCell sx={{ whiteSpace: 'nowrap' }}>
          {phoneNumber ? phoneNumber : 'N/A'}
        </TableCell>

        {/* Display Users with Avatar */}
        <TableCell sx={{ display: 'flex', alignItems: 'center' }}>
          <AvatarGroup
            max={4}
            sx={{
              display: 'inline-flex',
              [`& .${avatarGroupClasses.avatar}`]: {
                width: 24,
                height: 24,
                '&:first-of-type': {
                  fontSize: 12,
                },
              },
            }}
          >
           {users && users.length > 0 ? (
  users.map((user: { id: string; firstName: string; picture: string }) => (
    <Avatar key={user.id} alt={user.firstName} src={user.picture} />
  ))
) : (
  <Avatar alt="No Users" />
            )}
          </AvatarGroup>
        </TableCell>

        <TableCell align="right" sx={{ px: 1 }}>
          <IconButton color={popover.open ? 'inherit' : 'default'} onClick={popover.onOpen}>
            <Iconify icon="eva:more-vertical-fill" />
          </IconButton>
        </TableCell>
      </TableRow>

      <CustomPopover
        open={popover.open}
        onClose={popover.onClose}
        arrow="right-top"
        sx={{ width: 160 }}
      >
        <MenuItem
          onClick={() => {
            onViewRow();
            popover.onClose();
          }}
        >
          <Iconify icon="solar:eye-bold" />
          View
        </MenuItem>

        <MenuItem
          onClick={() => {
            onEditRow();
            popover.onClose();
          }}
        >
          <Iconify icon="solar:pen-bold" />
          Edit
        </MenuItem>

        <Divider sx={{ borderStyle: 'dashed' }} />

        <MenuItem
          onClick={() => {
            confirm.onTrue();
            popover.onClose();
          }}
          sx={{ color: 'error.main' }}
        >
          <Iconify icon="solar:trash-bin-trash-bold" />
          Delete
        </MenuItem>
      </CustomPopover>

      <ConfirmDialog
        open={confirm.value}
        onClose={confirm.onFalse}
        title="Delete"
        content="Are you sure want to delete?"
        action={
          <Button variant="contained" color="error" onClick={onDeleteRow}>
            Delete
          </Button>
        }
      />
    </>
  );
}
