import uniq from 'lodash/uniq';
import flatten from 'lodash/flatten';
import { useEffect, useCallback } from 'react';

import Box from '@mui/material/Box';
import Stack from '@mui/material/Stack';
import { useTheme } from '@mui/material/styles';
import IconButton from '@mui/material/IconButton';

import { useResponsive } from '@/hooks/use-responsive';

import Iconify from '@/shared/components/iconify';

import { IChatParticipant, IChatConversation, IClient, IChatConversationClient } from '@/shared/types/chat';

import { useBoolean, useCollapseNav } from '@/hooks';
import ChatRoomGroup from './chat-room-group';
import ChatRoomSingle from './chat-room-single';
import ChatRoomAttachments from './chat-room-attachments';
import Scrollbar from '@/components/scrollbar';
import { Collapse, ListItemButton, Typography } from '@mui/material';

// ----------------------------------------------------------------------

const NAV_WIDTH = 340;

type Props = {
  participants: IClient[];
  conversation: IChatConversationClient;
  children:React.ReactNode;
};

export default function ChatRoom({ participants, conversation,children }: Props) {
  const theme = useTheme();

  const lgUp = useResponsive('up', 'lg');
  const mdUp = useResponsive('up', 'md');

  const collapse = useBoolean(true);


  const {
    collapseDesktop,
    onCloseDesktop,
    onCollapseDesktop,
    //
    openMobile,
    onOpenMobile,
    onCloseMobile,
  } = useCollapseNav();

  const handleToggleNav = useCallback(() => {
    if (lgUp) {
      onCollapseDesktop();
    } else {
      onOpenMobile();
    }
    // eslint-disable-next-line react-hooks/exhaustive-deps
  }, [lgUp]);

  const group = participants.length > 2;

  const attachments = uniq(flatten(conversation?.messages.map((messages) => messages.attachments)));
  
  const renderDescription = (
    <>
      <ListItemButton
        onClick={collapse.onToggle}
        sx={{
          pl: 2.5,
          pr: 1.5,
          mt:7,
          height: 40,
          flexShrink: 0,
          flexGrow: 'unset',
          typography: 'overline',
          color: 'text.secondary',
          bgcolor: 'background.neutral',
        }}
      >
        <Box component="span" sx={{ flexGrow: 1 }}>
          Description 
        </Box>
        <Iconify
          width={16}
          icon={collapse.value ? 'eva:arrow-ios-downward-fill' : 'eva:arrow-ios-forward-fill'}
        />
      </ListItemButton>
    </>
  );

  const renderContent = (
    <>
      
      {group ? (
        <>
          {renderDescription}
          <div>
            <Collapse in={collapse.value}>
              <Typography variant='body2' sx={{p:2}}>
              This channel allows you to receive exclusive news and information about marketing. 
              You will be informed of the latest trends, special offers and upcoming events.


              </Typography>
            </Collapse>
          </div>
          <ChatRoomGroup participants={participants} />
        </>
      ) : (
        <ChatRoomSingle participant={participants[0]} />
          
      )}
        <Scrollbar>
        <ChatRoomAttachments attachments={attachments} />
        </Scrollbar>
    </>
  );

  return (
    <Box sx={{ position: 'relative' ,width:mdUp?NAV_WIDTH:1}}>
      <Stack direction='row' alignItems='center' 
        sx={{ position: 'absolute' ,
              mt:0,left:0,width:1,
              zIndex:111,
              backgroundColor:theme.palette.background.paper,
              borderLeft: mdUp?`solid 1px ${theme.palette.divider}`:'',
              }}>
        {children}
      <Typography variant='subtitle1'> Contacts Infos </Typography>
      </Stack>
        <Stack
          sx={{
            height: 1,
            flexShrink: 0,
            width:1,
            borderLeft: mdUp?`solid 1px ${theme.palette.divider}`:'',
            transition: theme.transitions.create(['width'], {
              duration: theme.transitions.duration.shorter,
            }),
            ...(collapseDesktop && {
              width: 0,
            }),
          }}
        >
          <Scrollbar>
          {renderContent}
          </Scrollbar>
        </Stack>
    </Box>
  );
}
