import React, { useEffect } from "react";
import {
  Button,
  Card,
  Dialog,
  DialogActions,
  DialogContent,
  DialogTitle,
  Divider,
  Grid,
  Stack,
} from "@mui/material";
import { fDate } from "@/utils/format-time";
import { IAgency, IAgencyNotification } from "@/shared/types/agency";

type Props = {
  open: boolean;
  onClose: VoidFunction;
  agency: IAgency | IAgencyNotification;
};

export default function DialogAgency({ open, onClose, agency }: Props) {
  useEffect(() => {
    if (agency) {
      console.log("Agency Data:", agency);
      console.log("Agency Address:", agency?.address);
    } else {
      console.log("No Agency Data");
    }
  }, [agency]);

  return (
    <Dialog fullWidth maxWidth="xs" open={open} onClose={onClose}>
      <DialogTitle>Agency Information</DialogTitle>
      <DialogContent sx={{ overflow: "unset" }}>
        <Grid container spacing={2}>
          <Grid item xs={12}>
            <Card sx={{ height: "100%" }}>
              <Stack
                spacing={2}
                sx={{ p: 3, pt: 0, typography: "body2", mt: 5 }}
              >
                <InfoRow
                  label="Agency Name"
                  value={agency?.agencyName || "N/A"}
                />
                <Divider sx={{ borderStyle: "dashed" }} />

                {/* Agency Description with raw HTML rendering */}
                <Grid container spacing={{ xs: 0.5, md: 2 }}>
                  <Grid
                    item
                    xs={12}
                    md={4}
                    sx={{
                      typography: "subtitle2",
                      textTransform: "capitalize",
                    }}
                  >
                    Agency Description
                  </Grid>
                  <Grid item xs={12} md={12} sx={{ color: "text.secondary" }}>
                    {agency?.description ? (
                      <div
                        dangerouslySetInnerHTML={{ __html: agency.description }}
                      />
                    ) : (
                      "N/A"
                    )}
                  </Grid>
                </Grid>
                <Divider sx={{ borderStyle: "dashed" }} />

                <InfoRow
                  label="Agency Phone Number"
                  value={agency?.phoneNumber || "N/A"}
                />
                <Divider sx={{ borderStyle: "dashed" }} />

                <InfoRow
                  label="Create Date"
                  value={agency?.createDate 
                    ? fDate(Array.isArray(agency.createDate) 
                        ? convertDateArray(agency.createDate)
                        : agency.createDate) 
                    : "N/A"}
                />
                <Divider sx={{ borderStyle: "dashed" }} />

                <InfoRow
                  label="Agency Address"
                  value={formatAddress(agency?.address)}
                />
                <Divider sx={{ borderStyle: "dashed" }} />
              </Stack>
            </Card>
          </Grid>
        </Grid>
      </DialogContent>
      <DialogActions sx={{ justifyContent: "space-between" }}>
        <Button variant="outlined" color="inherit" onClick={onClose}>
          Close
        </Button>
      </DialogActions>
    </Dialog>
  );
}

// Helper component for displaying a row of information
function InfoRow({ label, value }: { label: string; value?: string | null }) {
  return (
    <Grid container spacing={{ xs: 0.5, md: 2 }}>
      <Grid
        item
        xs={12}
        md={4}
        sx={{ typography: "subtitle2", textTransform: "capitalize" }}
      >
        {label}
      </Grid>
      <Grid item xs={12} md={12} sx={{ color: "text.secondary" }}>
        {value || "N/A"}
      </Grid>
    </Grid>
  );
}

// Helper function to format the address object
function formatAddress(address?: {
  primaryAddresse?: string;
  optionalAddress?: string;
  city?: string;
  postalCode?: string;
  country?: string;
}): string {
  if (!address) return "N/A";
  const { primaryAddresse, optionalAddress, city, postalCode, country } =
    address;
  return [primaryAddresse, optionalAddress, city, postalCode, country]
    .filter(Boolean)
    .join(", ");
}

function convertDateArray(dateArray: number[]): Date {
  const [year, month, day, hour, minute, second] = dateArray;
  return new Date(Date.UTC(year, month - 1, day, hour, minute, second));
}
