import { useState } from 'react';
import {
  Box,
  Menu,
  Switch,
  Button,
  Divider,
  MenuItem,
  Typography,
} from '@mui/material';
import Iconify from '@/shared/components/iconify/iconify';

interface DisplaySettings {
  showPriority: boolean;
  showDueDate: boolean;
  showType: boolean;
  showTags: boolean;
  showPoints: boolean;
}

type Props = {
  settings: DisplaySettings;
  onSettingsChange: (newSettings: DisplaySettings) => void;
};

export default function KanbanDisplaySettings({ settings, onSettingsChange }: Props) {
  const [anchorEl, setAnchorEl] = useState<null | HTMLElement>(null);

  const handleClick = (event: React.MouseEvent<HTMLElement>) => {
    setAnchorEl(event.currentTarget);
  };

  const handleClose = () => {
    setAnchorEl(null);
  };

  const handleToggle = (key: keyof DisplaySettings) => {
    onSettingsChange({
      ...settings,
      [key]: !settings[key],
    });
  };

  return (
    <>
      <Button
        color="inherit"
        onClick={handleClick}
        endIcon={<Iconify icon="eva:settings-2-outline" />}
      >
        Display Options
      </Button>

      <Menu
        anchorEl={anchorEl}
        open={Boolean(anchorEl)}
        onClose={handleClose}
        PaperProps={{
          sx: { width: 220, maxWidth: '100%' },
        }}
      >
        <Box sx={{ p: 2 }}>
          <Typography variant="subtitle2">Show Task Details</Typography>
        </Box>

        <Divider />

        <MenuItem sx={{ py: 1 }}>
          <Box sx={{ width: 1, display: 'flex', alignItems: 'center' }}>
            <Typography variant="body2">Priority</Typography>
            <Box sx={{ flexGrow: 1 }} />
            <Switch
              checked={settings.showPriority}
              onChange={() => handleToggle('showPriority')}
            />
          </Box>
        </MenuItem>

        <MenuItem sx={{ py: 1 }}>
          <Box sx={{ width: 1, display: 'flex', alignItems: 'center' }}>
            <Typography variant="body2">Due Date</Typography>
            <Box sx={{ flexGrow: 1 }} />
            <Switch
              checked={settings.showDueDate}
              onChange={() => handleToggle('showDueDate')}
            />
          </Box>
        </MenuItem>

        <MenuItem sx={{ py: 1 }}>
          <Box sx={{ width: 1, display: 'flex', alignItems: 'center' }}>
            <Typography variant="body2">Type</Typography>
            <Box sx={{ flexGrow: 1 }} />
            <Switch
              checked={settings.showType}
              onChange={() => handleToggle('showType')}
            />
          </Box>
        </MenuItem>

        <MenuItem sx={{ py: 1 }}>
          <Box sx={{ width: 1, display: 'flex', alignItems: 'center' }}>
            <Typography variant="body2">Tags</Typography>
            <Box sx={{ flexGrow: 1 }} />
            <Switch
              checked={settings.showTags}
              onChange={() => handleToggle('showTags')}
            />
          </Box>
        </MenuItem>

        <MenuItem sx={{ py: 1 }}>
          <Box sx={{ width: 1, display: 'flex', alignItems: 'center' }}>
            <Typography variant="body2">Points</Typography>
            <Box sx={{ flexGrow: 1 }} />
            <Switch
              checked={settings.showPoints}
              onChange={() => handleToggle('showPoints')}
            />
          </Box>
        </MenuItem>
      </Menu>
    </>
  );
} 